-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles myself, the author, to unlimited free and unrestricted use, access, and distribution of any works related to the unauthorized user and/or distributor.

--]]

THGuiTopDownCamera = {
    DEFAULTS = {
        DISTANCE_MIN_Z = GuiTopDownCamera.DISTANCE_MIN_Z,
        MOVE_SPEED_FACTOR_NEAR = GuiTopDownCamera.MOVE_SPEED_FACTOR_NEAR,
        MOVE_SPEED_FACTOR_FAR = GuiTopDownCamera.MOVE_SPEED_FACTOR_FAR,
        CAMERA_ZOOM_FACTOR = GuiTopDownCamera.CAMERA_ZOOM_FACTOR,
        CAMERA_ZOOM_FACTOR_MIN = GuiTopDownCamera.CAMERA_ZOOM_FACTOR_MIN,
        COLLISION_MASK = GuiTopDownCamera.COLLISION_MASK
    }
}
THGuiTopDownCamera.activate = function()
    THUtils.pcall(function()
        GuiTopDownCamera.DISTANCE_MIN_Z = 0
        GuiTopDownCamera.MOVE_SPEED_FACTOR_NEAR = 0.25
        GuiTopDownCamera.MOVE_SPEED_FACTOR_FAR = 32
        GuiTopDownCamera.CAMERA_ZOOM_FACTOR = 0.01
        GuiTopDownCamera.CAMERA_ZOOM_FACTOR_MIN = 0.01
        GuiTopDownCamera.COLLISION_MASK = CollisionFlag.DEFAULT
    end)
end
THGuiTopDownCamera.deactivate = function()
    THUtils.pcall(function()
        for key, value in pairs(THGuiTopDownCamera.DEFAULTS) do
            GuiTopDownCamera[key] = value
        end
    end)
end
THConstructionScreen = {}
local THConstructionScreen_mt = Class(THConstructionScreen)
THConstructionScreen.CLASS_NAME = "THConstructionScreen"
THConstructionScreen.DATA_KEY = tostring(THConstructionScreen)
THConstructionScreen.ACTION_TEXT = {
    SHOW_CONFIGS = g_i18n:getText("input_CONSTRUCTION_SHOW_CONFIGS"),
    PRECISION_MODE = g_i18n:getText("input_TH_DESIGN_KIT_PRECISION_MODE"),
    PRECISION_MODE_ON = g_i18n:getText("thAction_precisionModeOn"),
    PRECISION_MODE_OFF = g_i18n:getText("thAction_precisionModeOff")
}
function THConstructionScreen.getCustomData(target)
    return THUtils.call(function()
        return THUtils.getDataTable(target, THConstructionScreen.DATA_KEY)
    end)
end
THPlaceableEditBrush = {}
THPlaceableEditBrush.CLASS_NAME = "THPlaceableEditBrush"
THPlaceableEditBrush.DATA_KEY = tostring(THPlaceableEditBrush)
function THPlaceableEditBrush.getCustomData(target)
    return THUtils.call(function()
        return THUtils.getDataTable(target, THPlaceableEditBrush.DATA_KEY)
    end)
end
local function initScript()
    THUtils.prependFunction("ConstructionScreen", "setBrush", false, false, nil, THConstructionScreen.gPrepend_setBrush)
end
function THConstructionScreen.new(parent, customMt)
    customMt = customMt or THConstructionScreen_mt
    if THUtils.argIsValid(THUtils.getIsType(parent, ConstructionScreen), "parent", parent)
        and THUtils.argIsValid(type(customMt) == THValueType.TABLE, "customMt", customMt)
    then
        local self = THUtils.createDataTable(parent, THConstructionScreen.DATA_KEY, customMt)
        if self ~= nil then
            self.isServer = g_thDesignKit.coreData.isServer == true
            self.isClient = g_thDesignKit.coreData.isClient == true
            self.actionEventIds = {}
            self.dialogTexts = {
                purchaseConfig = g_i18n:getText("thDialogText_purchaseConfig")
            }
            self.dialogTitles = {
                purchaseConfig = g_i18n:getText("thDialogTitle_purchaseConfig")
            }
            self.requiredPermissions = {
                editProperty = {
                    [Farm.PERMISSION.SELL_PLACEABLE] = true
                }
            }
            self.isPrecisionModeActive = false
            local camera = parent.camera
            if camera ~= nil then
                local cameraData = THUtils.getDataTable(camera, THConstructionScreen.DATA_KEY)
                if cameraData == nil then
                    cameraData = THUtils.createDataTable(camera, THConstructionScreen.DATA_KEY)
                end
                if cameraData ~= nil then
                    cameraData.isEdgeScrollLocked = false
                end
            end
            if not self.areHooksCreated then
                THUtils.prependFunction(parent, "onClose", false, false, self, THConstructionScreen.prepend_onClose)
                THUtils.appendFunction(parent, "onShowConfigs", false, false, self, THConstructionScreen.append_onShowConfigs)
                THUtils.appendFunction(parent, "registerBrushActionEvents", false, false, self, THConstructionScreen.append_registerBrushActionEvents)
                THUtils.appendFunction(parent, "removeBrushActionEvents", false, false, self, THConstructionScreen.append_removeBrushActionEvents)
                THUtils.appendFunction(parent, "updateBrushActionTexts", false, false, self, THConstructionScreen.append_updateBrushActionTexts)
                THUtils.overwriteFunction(parent, "update", false, false, self, THConstructionScreen.overwrite_update)
                THUtils.overwriteFunction(parent, "mouseEvent", false, false, self, THConstructionScreen.overwrite_mouseEvent)
                THUtils.overwriteFunction(parent, "loadCurrentConfiguration", false, false, self, THConstructionScreen.overwrite_loadCurrentConfiguration)
                THUtils.overwriteFunction(parent, "processStoreItemConfigurations", false, false, self, THConstructionScreen.overwrite_processStoreItemConfigurations)
                if camera ~= nil and camera.setMouseEdgeScrollingActive ~= nil then
                    THUtils.overwriteFunction(camera, "setMouseEdgeScrollingActive", false, false, self, THConstructionScreen.overwrite_setMouseEdgeScrollingActive)
                end
                self.areHooksCreated = true
            end
            return self
        end
    end
end
function THConstructionScreen.actionEditConfigs(self, ...)
    local screen = self.parent
    local allowEdit = false
    THUtils.pcall(function()
        local brush = screen.brush
        if brush ~= nil and brush == screen.selectorBrush then
            local placeable = brush.lastPlaceable
            if placeable ~= nil and placeable.storeItem ~= nil then
                local editBrush = self.placeableEditBrush
                if editBrush == nil then
                    editBrush = THPlaceableEditBrush.new(brush)
                    self.placeableEditBrush = editBrush
                end
                local brushData = THPlaceableEditBrush.getCustomData(editBrush)
                if brushData ~= nil and editBrush ~= nil
                    and self:getIsEditAllowed(placeable)
                then
                    local srcConfigs = placeable.configurations
                    local srcConfigData = placeable.configurationData
                    if srcConfigs ~= nil and next(srcConfigs) ~= nil then
                        local tgtConfigs = THUtils.copyTable(srcConfigs, 1)
                        local tgtConfigData = nil
                        local defaultValues = brushData.placeableDefaults
                        defaultValues.configurations = THUtils.copyTable(srcConfigs, 1)
                        defaultValues.configurationData = nil
                        if srcConfigData ~= nil then
                            tgtConfigData = THUtils.copyTable(srcConfigData, 3)
                            defaultValues.configurationData = THUtils.copyTable(srcConfigData, 3)
                        else
                            tgtConfigData = {}
                            defaultValues.configurationData = {}
                        end
                        editBrush.placeable = placeable
                        editBrush:setStoreItem(placeable.storeItem, tgtConfigs, tgtConfigData)
                        screen.destructMode = false
                        screen:setBrush(editBrush, true)
                        allowEdit = true
                    end
                end
            end
        end
    end)
    if allowEdit then
        return screen:onShowConfigs(...)
    end
end
function THConstructionScreen.actionTogglePrecisionMode(self, ...)
    local screen = self.parent
    if not self.isPrecisionModeActive
        and screen ~= nil and screen.camera ~= nil
        and screen.camera:getIsActive()
    then
        self.isPrecisionModeActive = true
        THGuiTopDownCamera.activate()
    else
        self.isPrecisionModeActive = false
        THGuiTopDownCamera.deactivate()
    end
    self:updateActionEvents()
end
function THConstructionScreen.registerActionEvents(self, noUpdate)
    self:removeActionEvents()
    noUpdate = THUtils.validateArg(not noUpdate or noUpdate == true, "noUpdate", noUpdate, false)
    local screen = self.parent
    local brush = screen.brush
    if brush ~= nil then
        local _, eventId = g_inputBinding:registerActionEvent(InputAction.TH_DESIGN_KIT_PRECISION_MODE, self, self.actionTogglePrecisionMode, false, true, false, true)
        if eventId ~= nil then
            g_inputBinding:setActionEventTextPriority(eventId, GS_PRIO_VERY_HIGH)
            g_inputBinding:setActionEventTextVisibility(eventId, false)
            g_inputBinding:setActionEventActive(eventId, false)
        end
        self.actionEventIds.precisionMode = eventId
        if brush == screen.selectorBrush then
            _, eventId = g_inputBinding:registerActionEvent(InputAction.TH_DESIGN_KIT_CUSTOMIZE, self, self.actionEditConfigs, false, true, false, true)
            if eventId ~= nil then
                g_inputBinding:setActionEventText(eventId, THConstructionScreen.ACTION_TEXT.SHOW_CONFIGS)
                g_inputBinding:setActionEventTextPriority(eventId, GS_PRIO_VERY_HIGH)
                g_inputBinding:setActionEventTextVisibility(eventId, false)
                g_inputBinding:setActionEventActive(eventId, false)
            end
            self.actionEventIds.editConfigs = eventId
        end
        if noUpdate ~= true then
            self:updateActionEvents()
        end
    end
end
function THConstructionScreen.removeActionEvents(self)
    for eventName, eventId in pairs(self.actionEventIds) do
        g_inputBinding:removeActionEvent(eventId)
        self.actionEventIds[eventName] = nil
    end
end
function THConstructionScreen.updateActionEvents(self)
    local mission = g_currentMission
    local screen = self.parent
    local brush = screen.brush
    for _, eventId in pairs(self.actionEventIds) do
        local isEventActive = false
        if eventId == self.actionEventIds.editConfigs then
            if mission ~= nil
                and brush ~= nil and brush == screen.selectorBrush
                and brush.lastPlaceable ~= nil
                and brush.lastPlaceable.configurations ~= nil
                and next(brush.lastPlaceable.configurations) ~= nil
                and self:getIsEditAllowed(brush.lastPlaceable)
            then
                isEventActive = true
            end
        elseif eventId == self.actionEventIds.precisionMode then
            if self.isPrecisionModeActive then
                g_inputBinding:setActionEventText(eventId, THConstructionScreen.ACTION_TEXT.PRECISION_MODE_OFF)
            else
                g_inputBinding:setActionEventText(eventId, THConstructionScreen.ACTION_TEXT.PRECISION_MODE_ON)
            end
            isEventActive = true
        end
        g_inputBinding:setActionEventTextVisibility(eventId, isEventActive)
        g_inputBinding:setActionEventActive(eventId, isEventActive)
    end
end
function THConstructionScreen.getIsEditAllowed(self, object, farmId)
    local mission = g_currentMission
    if THUtils.argIsValid(object == nil or THUtils.getIsType(object, Object), "object", object)
        and mission ~= nil
    then
        if THUtils.getIsMasterUser() then
            return true
        end
        local allowAccess = true
        for permissionId in pairs(self.requiredPermissions.editProperty) do
            if not mission:getHasPlayerPermission(permissionId) then
                allowAccess = false
                break
            end
        end
        if allowAccess and object ~= nil then
            farmId = farmId or mission:getFarmId()
            if not mission.accessHandler:canFarmAccess(farmId, object) then
                allowAccess = false
            end
        end
        return allowAccess
    end
    return false
end
function THConstructionScreen.setCurrentConfiguration(self, configs, configData, isPurchased, purchasePrice)
    isPurchased = THUtils.validateArg(not isPurchased or isPurchased == true, "isPurchased", isPurchased, false)
    local mission = g_currentMission
    local success = false
    if THUtils.argIsValid(configs == nil or type(configs) == THValueType.TABLE, "configs", configs)
        and THUtils.argIsValid(configData == nil or type(configData) == THValueType.TABLE, "configData", configData)
        and THUtils.argIsValid(purchasePrice == nil or (type(purchasePrice) == THValueType.NUMBER and purchasePrice >= 0), "purchasePrice", purchasePrice)
        and mission ~= nil
    then
        local screen = self.parent
        local brushData, brush = THPlaceableEditBrush.getCustomData(screen.brush)
        if brushData ~= nil and brush == self.placeableEditBrush then
            local placeable = brush.placeable
            if placeable ~= nil then
                configs = configs or placeable.configurations
                configData = configData or placeable.configurationData
                if configs ~= nil and configData ~= nil
                    and self:getIsEditAllowed(placeable)
                then
                    local playerFarmId = mission:getFarmId()
                    local _, isProcessed = nil, false
                    local failMsgText = nil
                    if isPurchased then
                        if purchasePrice == nil then
                            _, purchasePrice = mission.economyManager:getBuyPrice(placeable.storeItem, configs)
                            purchasePrice = math.max(0, purchasePrice or 0)
                        end
                    else
                        purchasePrice = 0
                    end
                    local function processChanges()
                        if g_thDesignKit:updatePlaceableConfigurations(placeable, configs, configData, nil, isPurchased, purchasePrice, playerFarmId) then
                            brush:setStoreItem(placeable.storeItem, configs, configData)
                            isProcessed = true
                        end
                    end
                    if isPurchased then
                        local currentMoney = mission:getMoney(playerFarmId) or 0
                        if purchasePrice <= 0 or (currentMoney - purchasePrice >= 0) then
                            processChanges()
                        else
                            failMsgText = g_i18n:getText("thMessage_notEnoughMoney")
                        end
                    else
                        processChanges()
                    end
                    if not isProcessed then
                        local defaultValues = brushData.placeableDefaults
                        configs = defaultValues.configurations or placeable.configurations
                        configData = defaultValues.configurationData or placeable.configurationData
                        isPurchased = true
                        purchasePrice = 0
                        processChanges()
                        if failMsgText ~= nil then
                            mission:showBlinkingWarning(failMsgText, 2000)
                        end
                    end
                    success = isProcessed == true
                end
            end
        end
    end
    return success
end
function THConstructionScreen.prepend_onClose(self, superFunc, screen, ...)
    self.isPrecisionModeActive = false
    THGuiTopDownCamera.deactivate()
end
function THConstructionScreen.append_onShowConfigs(self, superFunc, screen, ...)
    if screen.configsBox:getIsVisible() then
        self:registerActionEvents()
    end
end
function THConstructionScreen.append_registerBrushActionEvents(self, superFunc, screen, ...)
    self:registerActionEvents()
end
function THConstructionScreen.append_removeBrushActionEvents(self, superFunc, screen, ...)
    self:removeActionEvents()
end
function THConstructionScreen.append_updateBrushActionTexts(self, superFunc, screen, ...)
    self:updateActionEvents()
end
function THConstructionScreen.overwrite_update(self, superFunc, screen, dt, ...)
    local function appendFunc(...)
        THUtils.pcall(function()
            local isConfigsBoxVisible = screen.configsBox:getIsVisible()
            if not isConfigsBoxVisible then
                local brushData, brush = THPlaceableEditBrush.getCustomData(screen.brush)
                local function deactivateBrush()
                    brush.placeable = nil
                    screen:setBrush(brushData.parent, true)
                end
                if brushData ~= nil and brush ~= nil and brush == self.placeableEditBrush then
                    if brush.placeable ~= nil then
                        local function restoreDefaults()
                            local defaultValues = brushData.placeableDefaults
                            self:setCurrentConfiguration(defaultValues.configurations, defaultValues.configurationData, true, 0)
                        end
                        YesNoDialog.show(function(pIsYes)
                            THUtils.pcall(function()
                                if pIsYes then
                                    self:setCurrentConfiguration(nil, nil, true)
                                else
                                    restoreDefaults()
                                end
                                deactivateBrush()
                            end)
                        end, nil, self.dialogTexts.purchaseConfig, self.dialogTitles.purchaseConfig)
                    else
                        deactivateBrush()
                    end
                end
            end
        end)
        return ...
    end
    return appendFunc(superFunc(screen, dt, ...))
end
function THConstructionScreen.overwrite_mouseEvent(self, superFunc, screen, ...)
    local function appendFunc(...)
        THUtils.pcall(function()
            if screen.camera ~= nil and screen.camera.setMouseEdgeScrollingActive ~= nil then
                local cameraData = THUtils.getDataTable(screen.camera, THConstructionScreen.DATA_KEY)
                if cameraData ~= nil then
                    local isConfigsBoxVisible = screen.configsBox:getIsVisible()
                    if isConfigsBoxVisible and self.isPrecisionModeActive then
                        if not cameraData.isEdgeScrollLocked then
                            cameraData.lastIsEdgeScrollActive = screen.camera.isMouseEdgeScrollingActive
                            screen.camera:setMouseEdgeScrollingActive(false)
                            cameraData.isEdgeScrollLocked = true
                        end
                    else
                        if cameraData.isEdgeScrollLocked then
                            cameraData.isEdgeScrollLocked = false
                            if cameraData.lastIsEdgeScrollActive ~= nil then
                                screen.camera:setMouseEdgeScrollingActive(cameraData.lastIsEdgeScrollActive)
                                cameraData.lastIsEdgeScrollActive = nil
                            end
                        end
                    end
                end
            end
        end)
        return ...
    end
    return appendFunc(superFunc(screen, ...))
end
function THConstructionScreen.overwrite_loadCurrentConfiguration(self, superFunc, screen, storeItem, ...)
    local prependSuccess = false
    THUtils.pcall(function()
        local brushData, brush = THPlaceableEditBrush.getCustomData(screen.brush)
        if brushData ~= nil and brush ~= nil and brush == self.placeableEditBrush then
            if screen.configurations ~= nil and screen.configurationData ~= nil then
                self:setCurrentConfiguration(screen.configurations, screen.configurationData)
            end
            prependSuccess = true
        end
    end)
    if not prependSuccess then
        return superFunc(screen, storeItem, ...)
    end
end
function THConstructionScreen.overwrite_processStoreItemConfigurations(self, superFunc, screen, storeItem, ...)
    THUtils.pcall(function()
        local brushData, brush = THPlaceableEditBrush.getCustomData(screen.brush)
        if brushData ~= nil and brush ~= nil and brush == self.placeableEditBrush then
            if brush.configurationData ~= nil then
                screen.configurationData = THUtils.copyTable(brush.configurationData, 3)
            end
        end
    end)
    return superFunc(screen, storeItem, ...)
end
function THConstructionScreen.overwrite_setMouseEdgeScrollingActive(_, superFunc, camera, isActive, ...)
    local allowUpdate = true
    THUtils.pcall(function()
        local cameraData = THUtils.getDataTable(camera, THConstructionScreen.DATA_KEY)
        if cameraData ~= nil then
            if cameraData.isEdgeScrollLocked then
                if isActive ~= nil then
                    cameraData.lastIsEdgeScrollActive = isActive
                end
                allowUpdate = false
            else
                cameraData.lastIsEdgeScrollActive = nil
            end
        end
    end)
    if allowUpdate then
        return superFunc(camera, isActive, ...)
    end
end
function THConstructionScreen.gPrepend_setBrush(superFunc, screen, brush, ...)
    local self = THConstructionScreen.getCustomData(screen)
    if self == nil then
        self = THConstructionScreen.new(screen)
    end
end
function THPlaceableEditBrush.new(parent)
    if THUtils.argIsValid(THUtils.getIsType(parent, ConstructionBrushSelect), "parent", parent) then
        local self = setmetatable({}, { __index = parent })
        local customData = THUtils.createDataTable(self, THPlaceableEditBrush.DATA_KEY)
        customData.parent = parent
        customData.isEditBrush = true
        customData.placeableDefaults = {}
        return self
    end
end
THUtils.pcall(initScript)