-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles myself, the author, to unlimited free and unrestricted use, access, and distribution of any works related to the unauthorized user and/or distributor.

--]]

local thModName = g_currentModName
local thModPath = g_currentModDirectory
_G.THSpecType = {
    VEHICLE = 1,
    PLACEABLE = 2
}
THSpecManager = {}
local THSpecManager_mt = Class(THSpecManager)
THSpecManager.CLASS_NAME = "THSpecManager"
local function initScript()
    local self = THSpecManager.new()
    if self ~= nil then
        _G.g_thSpecManager = self
    end
end
function THSpecManager.new(customMt)
    customMt = customMt or THSpecManager_mt
    if THUtils.argIsValid(type(customMt) == THValueType.TABLE, "customMt", customMt) then
        local self = setmetatable({}, customMt)
        self.specTypes = {
            byName = {},
            byIndex = {}
        }
        self.specializations = {}
        for typeKey, typeIndex in pairs(THSpecType) do
            local specTypeData = { index = typeIndex }
            local isValidType = true
            if typeIndex == THSpecType.VEHICLE then
                specTypeData.name = "vehicle"
                specTypeData.namePlural = "vehicles"
                specTypeData.manager = g_specializationManager
            elseif typeIndex == THSpecType.PLACEABLE then
                specTypeData.name = "placeable"
                specTypeData.namePlural = "placeables"
                specTypeData.manager = g_placeableSpecializationManager
            else
                THUtils.errorMsg(true, THMessage.INVALID_VALUE, "specType", typeKey)
                isValidType = false
            end
            if isValidType then
                self.specTypes.byName[specTypeData.name:upper()] = specTypeData
                self.specTypes.byIndex[specTypeData.index] = specTypeData
                self.specializations[specTypeData.index] = {}
            end
        end
        if g_thEventManager ~= nil then
            g_thEventManager:addEventListener(self)
        end
        return self
    end
end
function THSpecManager.onFinalizeTypes(self, manager)
    local specTypeData = self:getSpecType(manager.typeName)
    if specTypeData ~= nil then
        local specsArray = self.specializations[specTypeData.index]
        local typedObjects = manager:getTypes()
        if specsArray ~= nil and typedObjects ~= nil then
            for specIndex = 1, #specsArray do
                local specData = specsArray[specIndex]
                local timesAdded = 0
                for objectTypeName, objectType in pairs(typedObjects) do
                    if objectType.specializations ~= nil
                        and not SpecializationUtil.hasSpecialization(specData.class, objectType.specializations)
                        and specData.class.prerequisitesPresent(objectType.specializations)
                    then
                        manager:addSpecialization(objectTypeName, specData.absName)
                        if SpecializationUtil.hasSpecialization(specData.class, objectType.specializations) then
                            timesAdded = timesAdded + 1
                        end
                    end
                end
                if timesAdded > 0 then
                    THUtils.displayMsg("Specialization %q added to %d %s", specData.name, timesAdded, specTypeData.namePlural)
                end
            end
        end
    end
end
function THSpecManager.getSpecType(self, specTypeId, verbose)
    local idVarType = type(specTypeId)
    local specTypeData = nil
    verbose = THUtils.validateArg(not verbose or verbose == true, "verbose", verbose, true)
    if idVarType == THValueType.STRING then
        specTypeData = self.specTypes.byName[specTypeId:upper()]
    elseif idVarType == THValueType.NUMBER then
        specTypeData = self.specTypes.byIndex[specTypeId]
    elseif specTypeId ~= nil then
        verbose = true
    end
    if specTypeData == nil and verbose then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "specTypeId", specTypeId)
    end
    return specTypeData
end
function THSpecManager.addSpecialization(self, specTypeId, specName, className, filename)
    local specTypeData = self:getSpecType(specTypeId)
    if THUtils.argIsValid(specTypeData ~= nil, "specTypeId", specTypeId)
        and THUtils.argIsValid(type(specName) == THValueType.STRING and specName ~= "", "specName", specName)
        and THUtils.argIsValid(type(className) == THValueType.STRING, "className", className)
        and THUtils.argIsValid(type(filename) == THValueType.STRING, "filename", filename)
    then
        local manager = specTypeData.manager
        local absSpecName = thModName .. "." .. specName
        local absClassName = thModPath .. "." .. className
        local absFilename = Utils.getFilename(filename, thModPath)
        if absFilename == nil or not fileExists(absFilename) then
            THUtils.errorMsg(nil, THMessage.FILE_NOT_FOUND, absFilename)
        elseif manager:getSpecializationByName(specName) then
            THUtils.errorMsg(nil, "Duplicate %s specialization found: %s", specTypeData.name, specName)
        else
            local specData = {
                name = specName,
                absName = absSpecName,
                className = className,
                absClassName = absClassName,
                filename = filename,
                absFilename = absFilename,
                specType = specTypeData,
                modName = thModName
            }
            manager:addSpecialization(specName, className, absFilename, thModName)
            if manager:getSpecializationByName(specName) then
                local specClass = _G[className]
                if specClass == nil then
                    THUtils.errorMsg(nil, "Cannot find specialization %q class table: %s", specName, className)
                else
                    specData.class = specClass
                    table.insert(self.specializations[specTypeData.index], specData)
                    THUtils.displayMsg("Added %s specialization: %s", specTypeData.name, specName)
                    return true
                end
            end
        end
    end
    return false
end
THUtils.pcall(initScript)