-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles
myself, the author, to unlimited free and unrestricted use,
access, and distribution of any works related to the unauthorized
user and/or distributor.

--]]

THDensityMapUpdater = {}
local THDensityMapUpdater_mt = Class(THDensityMapUpdater)
THDensityMapUpdater.CLASS_NAME = "THDensityMapUpdater"
THDensityMapUpdater.FUNCTION_CACHE_ID = {
    UPDATE_SOWING_AREA = "FSDENSITYMAPUTIL.UPDATESOWINGAREA",
    UPDATE_DIRECT_SOWING_AREA = "FSDENSITYMAPUTIL.UPDATEDIRECTSOWINGAREA",
    CUT_FRUIT_AREA = "FSDENSITYMAPUTIL.CUTFRUITAREA",
    UPDATE_DISC_HARROW_AREA = "FSDENSITYMAPUTIL.UPDATEDISCHARROWAREA",
    UPDATE_HERBICIDE_AREA = "FSDENSITYMAPUTIL.UPDATEHERBICIDEAREA"
}
THDensityMapUpdater.DEBUG_FLAG = {
    UPDATE_SOWING_AREA = "updateSowingArea"
}
local debugFlagId = THUtils.createDebugFlagId(THDensityMapUpdater.CLASS_NAME)
THDensityMapUpdater.debugFlagId = debugFlagId
local function initScript()
    local self = THDensityMapUpdater.new()
    if self ~= nil then
        _G.g_thDensityMapUpdater = self
    end
end
function THDensityMapUpdater.new(customMt)
    customMt = customMt or THDensityMapUpdater_mt
    local self = setmetatable({}, customMt)
    for _, otherDebugFlagId in pairs(THDensityMapUpdater.DEBUG_FLAG) do
        THUtils.createDebugFlagId(otherDebugFlagId)
    end
    g_thEventManager:addEventListener(self)
    g_thDensityMapController:addEventListener(self)
    return self
end
function THDensityMapController.onSetMissionInfo(self, mission, missionInfo, missionDynamicInfo)
    THUtils.appendFunction("FSDensityMapUtil", "updateDestroyCommonArea", false, false, self, THDensityMapUpdater.append_updateDestroyCommonArea)
    THUtils.overwriteFunction(g_thGlobalEnv, "getDensityTypeIndexAtWorldPos", false, false, self, THDensityMapUpdater.overwrite_getDensityTypeIndexAtWorldPos)
    THUtils.overwriteFunction(g_thGlobalEnv, "getDensityStatesAtWorldPos", false, false, self, THDensityMapUpdater.overwrite_getDensityStatesAtWorldPos)
    THUtils.overwriteFunction("FSDensityMapUtil", "getFruitArea", false, false, self, THDensityMapUpdater.overwrite_getFruitArea)
    THUtils.overwriteFunction("FSDensityMapUtil", "updateSowingArea", false, false, self, THDensityMapUpdater.overwrite_updateSowingArea)
    THUtils.overwriteFunction("FSDensityMapUtil", "updateDirectSowingArea", false, false, self, THDensityMapUpdater.overwrite_updateDirectSowingArea)
    THUtils.overwriteFunction("FSDensityMapUtil", "cutFruitArea", false, false, self, THDensityMapUpdater.overwrite_cutFruitArea)
    THUtils.overwriteFunction("FSDensityMapUtil", "updateDiscHarrowArea", false, false, self, THDensityMapUpdater.overwrite_updateDiscHarrowArea)
    THUtils.overwriteFunction("FSDensityMapUtil", "updateFruitPreparerArea", false, false, self, THDensityMapUpdater.overwrite_updateFruitPreparerArea)
    THUtils.overwriteFunction("FSDensityMapUtil", "updateHerbicideArea", false, false, self, THDensityMapUpdater.overwrite_updateHerbicideArea)
end
function THDensityMapUpdater.dmOnModifierExecute(self, parent, dmFunctionData, superFunc, callingParams, returnParams)
    if dmFunctionData.id == THDensityMapUpdater.FUNCTION_CACHE_ID.UPDATE_SOWING_AREA
        or dmFunctionData.id == THDensityMapUpdater.FUNCTION_CACHE_ID.UPDATE_DIRECT_SOWING_AREA
    then
        self:dmOnModifierExecuteSowingArea(parent, dmFunctionData, superFunc, callingParams, returnParams)
    elseif dmFunctionData.id == THDensityMapUpdater.FUNCTION_CACHE_ID.CUT_FRUIT_AREA then
        self:dmOnModifierExecuteCutFruitArea(parent, dmFunctionData, superFunc, callingParams, returnParams)
    end
end
function THDensityMapUpdater.dmOnMultiModifierAddExecute(self, parent, dmFunctionData, superFunc, callingParams, returnParams)
    if dmFunctionData.id == THDensityMapUpdater.FUNCTION_CACHE_ID.UPDATE_DIRECT_SOWING_AREA then
        self:dmOnMultiModifierAddExecuteSowingArea(parent, dmFunctionData, superFunc, callingParams, returnParams)
    elseif dmFunctionData.id == THDensityMapUpdater.FUNCTION_CACHE_ID.UPDATE_HERBICIDE_AREA then
        self:dmOnMultiModifierAddExecuteHerbicideArea(parent, dmFunctionData, superFunc, callingParams, returnParams)
    end
end
function THDensityMapUpdater.dmOnModifierExecuteSowingArea(self, parent, dmFunctionData, superFunc, callingParams, returnParams)
    if g_thRowCropSystem.infoLayers.isValid then
        local thSpacingLayer = g_thRowCropSystem.infoLayers.rowSpacing
        local dmSpacingFilter = thSpacingLayer.dmFilter
        local spacingValues = dmFunctionData.spacingValues
        local functionCache = dmFunctionData.functionCache
        if functionCache == nil then
            functionCache = FSDensityMapUtil.functionCache[dmFunctionData.funcName]
            if functionCache ~= nil then
                dmFunctionData.functionCache = functionCache
            end
        end
        if spacingValues ~= nil and functionCache ~= nil
            and functionCache.fruitModifiers ~= nil and functionCache.fruitFilters ~= nil
        then
            local rcsFruitData = spacingValues.fruitType
            local dmFuncType = callingParams.dmFuncType
            local dmBaseFruitModifier = functionCache.fruitModifiers[rcsFruitData.fruitType.index]
            local dmBaseGroundModifier = functionCache.groundTypeModifier
            local dmBaseGroundAngleModifier = functionCache.groundAngleModifier
            local dmCallingValue = callingParams.value
            if dmFuncType == THDMFuncType.SET then
                if parent == dmBaseFruitModifier then
                    if dmCallingValue > 0 then
                        g_thRowCropSystem:dmExecuteAreaMask(dmFunctionData, callingParams, thSpacingLayer.dmRowFilter2)
                    else
                        dmSpacingFilter:setValueCompareParams(DensityValueCompareType.GREATER, 0)
                        g_thRowCropSystem:dmExecuteAreaMask(dmFunctionData, callingParams, dmSpacingFilter)
                    end
                elseif parent == dmBaseGroundModifier
                    or parent == dmBaseGroundAngleModifier
                then
                    dmSpacingFilter:setValueCompareParams(DensityValueCompareType.GREATER, 0)
                    g_thRowCropSystem:dmExecuteAreaMask(dmFunctionData, callingParams, dmSpacingFilter)
                end
            end
        end
    end
end
function THDensityMapUpdater.dmOnMultiModifierAddExecuteSowingArea(self, parent, dmFunctionData, superFunc, callingParams, returnParams)
    if g_thRowCropSystem.infoLayers.isValid then
        local thSpacingLayer = g_thRowCropSystem.infoLayers.rowSpacing
        local dmSpacingFilter = thSpacingLayer.dmFilter
        local spacingValues = dmFunctionData.spacingValues
        local functionCache = dmFunctionData.functionCache
        if functionCache == nil then
            functionCache = FSDensityMapUtil.functionCache[dmFunctionData.funcName]
            if functionCache ~= nil then
                dmFunctionData.functionCache = functionCache
            end
        end
        if spacingValues ~= nil and functionCache ~= nil then
            local rcsFruitData = spacingValues.fruitType
            if functionCache.multiModifiers ~= nil and functionCache.multiModifiers[rcsFruitData.fruitType.index] ~= nil then
                local isMultiModifierValid = false
                local plantedState = nil
                for state, dmMultiModifier in pairs(functionCache.multiModifiers[rcsFruitData.fruitType.index]) do
                    if parent == dmMultiModifier then
                        isMultiModifierValid = true
                        plantedState = state
                        break
                    end
                end
                if isMultiModifierValid and plantedState ~= nil then
                    local dmFuncType = callingParams.dmFuncType
                    local dmCallingModifier = callingParams.dmModifier
                    local dmCallingValue = callingParams.value
                    local dmBaseGroundModifier = functionCache.groundTypeModifier
                    if dmFuncType == THDMFuncType.SET then
                        if dmCallingModifier == dmBaseGroundModifier then
                            dmSpacingFilter:setValueCompareParams(DensityValueCompareType.GREATER, 0)
                            g_thRowCropSystem:dmExecuteAreaMask(dmFunctionData, callingParams, dmSpacingFilter, true)
                        else
                            local dmModifierLayerId = g_thDensityMapController:getDensityMapLayerId(dmCallingModifier)
                            local thModifierFruitData = g_thMapTypeManager:getFruitTypeByDataPlaneId(dmModifierLayerId)
                            if thModifierFruitData ~= nil and functionCache.fruitModifiers ~= nil then
                                local dmBaseOtherFruitModifier = functionCache.fruitModifiers[thModifierFruitData.index]
                                if dmCallingModifier == dmBaseOtherFruitModifier then
                                    if dmCallingValue == 0 then
                                        dmSpacingFilter:setValueCompareParams(DensityValueCompareType.GREATER, 0)
                                        g_thRowCropSystem:dmExecuteAreaMask(dmFunctionData, callingParams, dmSpacingFilter, true)
                                    end
                                end
                            end
                        end
                    end
                end
            end
        end
    end
end
function THDensityMapUpdater.dmOnModifierExecuteCutFruitArea(self, parent, dmFunctionData, superFunc, callingParams, returnParams)
    if g_thRowCropSystem.infoLayers.isValid then
        local spacingValues = dmFunctionData.spacingValues
        local functionCache = dmFunctionData.functionCache
        if functionCache == nil then
            functionCache = FSDensityMapUtil.functionCache.cutFruitArea
            if functionCache ~= nil then
                dmFunctionData.functionCache = functionCache
            end
        end
        if spacingValues ~= nil and functionCache ~= nil and functionCache.fruitFilters ~= nil and functionCache.fruitValueModifiers ~= nil then
            local rcsFruitData = spacingValues.fruitType
            local dmFuncType = callingParams.dmFuncType
            local dmBaseFruitFilter = functionCache.fruitFilters[rcsFruitData.fruitType.index]
            local dmBaseSprayLevelModifier = functionCache.sprayLevelModifier
            local dmBaseLimeLevelModifier = functionCache.limeLevelModifier
            local dmBaseStubbleShredModifier = functionCache.stubbleShredModifier
            if dmFuncType == THDMFuncType.ADD then
                if parent == dmBaseLimeLevelModifier or parent == dmBaseStubbleShredModifier then
                    g_thRowCropSystem:dmExecuteHarvestLock(dmFunctionData, callingParams, dmBaseFruitFilter)
                end
            elseif dmFuncType == THDMFuncType.SET then
                if parent == dmBaseSprayLevelModifier then
                    g_thRowCropSystem:dmExecuteHarvestLock(dmFunctionData, callingParams, dmBaseFruitFilter)
                end
            end
        end
    end
end
function THDensityMapUpdater.dmOnMultiModifierAddExecuteHerbicideArea(self, parent, dmFunctionData, superFunc, callingParams, returnParams)
    local functionCache = FSDensityMapUtil.functionCache.updateHerbicideArea
    if g_thRowCropSystem.infoLayers.isValid and functionCache ~= nil then
        local dmGroundTypeLayerId = dmFunctionData.dmGroundTypeLayerId
        local dmCallingFilters = callingParams.dmFilters
        if dmCallingFilters ~= nil and dmCallingFilters.n > 0 then
            local groundFilterIndex = nil
            for filterIndex = 1, dmCallingFilters.n do
                local dmFilter = dmCallingFilters[filterIndex]
                local dmFilterLayerId = g_thDensityMapController:getDensityMapLayerId(dmFilter)
                if dmFilterLayerId ~= nil then
                    if dmGroundTypeLayerId ~= nil and dmFilterLayerId == dmGroundTypeLayerId then
                        groundFilterIndex = filterIndex
                    end
                end
            end
            if groundFilterIndex ~= nil then
                THUtils.removePackedEntry(dmCallingFilters, groundFilterIndex)
                if THUtils.getIsDebugEnabled(debugFlagId, THDebugLevel.UPDATE) then
                    THUtils.displayMsg("Allowing herbicide on all ground types")
                end
            end
        end
    end
end
function THDensityMapUpdater.append_updateDestroyCommonArea(self, superFunc, sx, sz, wx, wz, hx, hz, ...)
    if g_thRowCropSystem.infoLayers.isValid then
        local thRowShutoffLayer = g_thRowCropSystem.infoLayers.rowShutoff
        local thHarvestLockLayer = g_thRowCropSystem.infoLayers.harvestLock
        thRowShutoffLayer:resetArea(sx, sz, wx, wz, hx, hz)
        thHarvestLockLayer:resetArea(sx, sz, wx, wz, hx, hz)
    end
end
function THDensityMapUpdater.overwrite_getFruitArea(self, superFunc, fruitTypeIndex, sx, sz, wx, wz, hx, hz, ...)
    if fruitTypeIndex ~= nil and fruitTypeIndex ~= FruitType.UNKNOWN then
        THUtils.pcall(function()
            local spacingValues = g_thRowCropSystem:getFruitTypeRowSpacing(fruitTypeIndex)
            local fruitMetersPerPixel = g_thMapTypeManager:getFruitMetersPerPixel()
            if spacingValues ~= nil and fruitMetersPerPixel > 0 then
                local workWidth = THUtils.getVector2Distance(sx, sz, wx, wz)
                local spacingWidth = math.max(fruitMetersPerPixel, spacingValues.spacingWidth or 0)
                if workWidth > 0 and workWidth < spacingWidth then
                    local widthFactor = spacingWidth / workWidth
                    local mwx = (sx + wx) / 2
                    local mwz = (sz + wz) / 2
                    local dwx = (wx - sx) * widthFactor
                    local dwz = (wz - sz) * widthFactor
                    local dhx = hx - sx
                    local dhz = hz - sz
                    sx = mwx - (dwx * 0.5)
                    sz = mwz - (dwz * 0.5)
                    wx = sx + dwx
                    wz = sz + dwz
                    hx = sx + dhx
                    hz = sz + dhz
                end
            end
        end)
    end
    return superFunc(fruitTypeIndex, sx, sz, wx, wz, hx, hz, ...)
end
function THDensityMapUpdater.overwrite_getDensityTypeIndexAtWorldPos(self, superFunc, defaultDataPlaneId, x, y, z, ...)
    local function appendFunc(rDensityTypeIndex, ...)
        THUtils.pcall(function()
            if defaultDataPlaneId ~= g_fruitTypeManager:getDefaultDataPlaneId() then
                return
            end
            local rcsFruitData = g_thRowCropSystem:getRowCropFruitTypeAroundWorldPos(x, z)
            if rcsFruitData ~= nil and rcsFruitData.fruitType.desc.terrainDataPlaneIndex ~= nil then
                rDensityTypeIndex = rcsFruitData.fruitType.desc.terrainDataPlaneIndex
            end
        end)
        return rDensityTypeIndex, ...
    end
    return appendFunc(superFunc(defaultDataPlaneId, x, y, z, ...))
end
function THDensityMapUpdater.overwrite_getDensityStatesAtWorldPos(self, superFunc, defaultDataPlaneId, x, y, z, ...)
    local terrainNode = g_terrainNode
    local function appendFunc(rDensityStates, ...)
        if terrainNode ~= nil and terrainNode ~= 0 then
            THUtils.pcall(function()
                if defaultDataPlaneId ~= g_fruitTypeManager:getDefaultDataPlaneId() then
                    return
                end
                local rcsFruitData = g_thRowCropSystem:getRowCropFruitTypeAroundWorldPos(x, z)
                local fruitMetersPerPixel = g_thMapTypeManager:getFruitMetersPerPixel()
                if rcsFruitData ~= nil and fruitMetersPerPixel > 0 then
                    local spacingWidth = math.max(fruitMetersPerPixel, rcsFruitData.defaultValues.spacingWidth)
                    local startX = x - (spacingWidth * 0.5)
                    local startZ = z - (spacingWidth * 0.5)
                    local numSteps = THUtils.round(spacingWidth / fruitMetersPerPixel)
                    if numSteps > 0 then
                        for stepWidth = 1, numSteps do
                            local posX = startX + (fruitMetersPerPixel * (stepWidth - 1))
                            for stepLength = 1, numSteps do
                                local posZ = startZ + (fruitMetersPerPixel * (stepLength - 1))
                                local posY = getTerrainHeightAtWorldPos(terrainNode, posX, 0, posZ)
                                local newDensityStates = superFunc(defaultDataPlaneId, posX, posY, posZ)
                                if newDensityStates ~= nil then
                                    local growthState = rcsFruitData.fruitType.desc:getGrowthStateByDensityState(newDensityStates)
                                    if growthState ~= nil and growthState > 0 then
                                        rDensityStates = newDensityStates
                                        return
                                    end
                                end
                            end
                        end
                    end
                end
            end)
        end
        return rDensityStates, ...
    end
    return appendFunc(superFunc(defaultDataPlaneId, x, y, z, ...))
end
function THDensityMapUpdater.overwrite_updateSowingAreaShared(self, superFunc, funcName, fruitTypeIndex, sx, sz, wx, wz, hx, hz, groundType, useRidgeSeeding, angle, growthState, ...)
    local terrainNode = g_terrainNode
    local isDebugEnabled = false
    local prependSuccess = false
    if fruitTypeIndex ~= nil and fruitTypeIndex ~= FruitType.UNKNOWN then
        THUtils.pcall(function()
            local mission = g_thRowCropSystem.coreData.mission
            if THUtils.argIsValid(type(funcName) == THValueType.STRING, "funcName", funcName)
                and mission ~= nil and g_thRowCropSystem.infoLayers.isValid
                and terrainNode ~= nil and terrainNode ~= 0
            then
                local thAreaMaskLayer = g_thRowCropSystem.infoLayers.areaMask
                local dmAreaModifier = thAreaMaskLayer.dmModifier
                local functionCache = FSDensityMapUtil.functionCache[funcName]
                local dmFunctionData = THUtils.getFunctionCache(self, "FSDensityMapUtil", funcName)
                if dmFunctionData == nil then
                    dmFunctionData = THUtils.createFunctionCache(self, "FSDensityMapUtil", funcName)
                    dmFunctionData.rowCoords = {}
                end
                isDebugEnabled = THUtils.getIsDebugEnabled(THDensityMapUpdater.DEBUG_FLAG.UPDATE_SOWING_AREA)
                local spacingValues = g_thRowCropSystem:getFruitTypeRowSpacing(fruitTypeIndex)
                if spacingValues ~= nil then
                    local rcsFruitData = spacingValues.fruitType
                    dmFunctionData.superFunc = superFunc
                    dmFunctionData.functionCache = functionCache
                    dmFunctionData.fruitTypeIndex = rcsFruitData.fruitType.index
                    dmFunctionData.plantedState = growthState or 1
                    dmFunctionData.groundType = groundType
                    dmFunctionData.spacingValues = spacingValues
                    sx, sz, wx, wz, hx, hz = g_thRowCropSystem:getRowCropArea(sx, sz, wx, wz, hx, hz, 2, true)
                end
                if isDebugEnabled then
                    local tsy = getTerrainHeightAtWorldPos(terrainNode, sx, 0, sz)
                    local twy = getTerrainHeightAtWorldPos(terrainNode, wx, 0, wz)
                    local thy = getTerrainHeightAtWorldPos(terrainNode, hx, 0, hz)
                    DebugUtil.drawDebugAreaRectangle(sx, tsy + 0.5, sz, wx, twy + 0.5, wz, hx, thy + 0.5, hz, false, 1, 1, 1)
                    drawDebugLine(sx, tsy - 0.5, sz, 1, 0, 0, sx, tsy + 2, sz, 1, 0, 0)
                    drawDebugLine(wx, twy - 0.5, wz, 1, 0, 0, wx, twy + 2, wz, 1, 0, 0)
                    drawDebugLine(hx, thy - 0.5, hz, 1, 0, 0, hx, thy + 2, hz, 1, 0, 0)
                end
                if spacingValues ~= nil then
                    g_thRowCropSystem:updateRowSpacingArea(sx, sz, wx, wz, hx, hz, spacingValues, dmFunctionData.rowCoords, false, true)
                    dmAreaModifier:setParallelogramWorldCoords(sx, sz, wx, wz, hx, hz, DensityCoordType.POINT_POINT_POINT)
                    g_thDensityMapController:setActiveFunctionData(dmFunctionData)
                    prependSuccess = true
                end
            end
        end)
    end
    local function appendFunc(rChangedArea, rTotalArea, ...)
        if prependSuccess then
            THUtils.pcall(function()
                local dmFunctionData = THUtils.getFunctionCache(self, "FSDensityMapUtil", funcName)
                if dmFunctionData ~= nil then
                    if rChangedArea ~= nil and rChangedArea > 0 and rTotalArea ~= nil and rTotalArea > 0 then
                        local thRowShutoffLayer = g_thRowCropSystem.infoLayers.rowShutoff
                        local dmShutoffModifier = thRowShutoffLayer.dmModifier
                        local dmFieldFilter = g_thRowCropSystem.infoLayers.shared.dmFieldFilter
                        local spacingValues = dmFunctionData.spacingValues
                        local rcsFruitData = spacingValues.fruitType
                        rChangedArea = THUtils.clamp(rChangedArea * rcsFruitData.factors.seed, 0, rTotalArea)
                        g_thRowCropSystem.infoLayers.harvestLock:resetArea(sx, sz, wx, wz, hx, hz)
                        sx, sz, wx, wz, hx, hz = g_thRowCropSystem:getRowCropArea(sx, sz, wx, wz, hx, hz, 1)
                        dmShutoffModifier:setParallelogramWorldCoords(sx, sz, wx, wz, hx, hz, DensityCoordType.POINT_POINT_POINT)
                        local _, dmChangedArea = dmShutoffModifier:executeSetWithStats(1, dmFieldFilter)
                        if dmChangedArea ~= nil and dmChangedArea > 0 then
                            if isDebugEnabled then
                                local tsy = getTerrainHeightAtWorldPos(terrainNode, sx, 0, sz)
                                local twy = getTerrainHeightAtWorldPos(terrainNode, wx, 0, wz)
                                local thy = getTerrainHeightAtWorldPos(terrainNode, hx, 0, hz)
                                DebugUtil.drawDebugAreaRectangle(sx, tsy + 0.5, sz, wx, twy + 0.5, wz, hx, thy + 0.5, hz, false, 1, 0, 0)
                            end
                        end
                    end
                end
                g_thDensityMapController:restoreLastActiveFunctionData()
            end)
        end
        return rChangedArea, rTotalArea, ...
    end
    return appendFunc(superFunc(fruitTypeIndex, sx, sz, wx, wz, hx, hz, groundType, useRidgeSeeding, angle, growthState, ...))
end
function THDensityMapUpdater.overwrite_updateSowingArea(self, superFunc, fruitTypeIndex, sx, sz, wx, wz, hx, hz, groundType, useRidgeSeeding, angle, growthState, ...)
    return THDensityMapUpdater.overwrite_updateSowingAreaShared(self, superFunc, "updateSowingArea", fruitTypeIndex, sx, sz, wx, wz, hx, hz, groundType, useRidgeSeeding, angle, growthState, ...)
end
function THDensityMapUpdater.overwrite_updateDirectSowingArea(self, superFunc, fruitTypeIndex, sx, sz, wx, wz, hx, hz, groundType, useRidgeSeeding, angle, growthState, ...)
    return THDensityMapUpdater.overwrite_updateSowingAreaShared(self, superFunc, "updateDirectSowingArea", fruitTypeIndex, sx, sz, wx, wz, hx, hz, groundType, useRidgeSeeding, angle, growthState, ...)
end
function THDensityMapUpdater.overwrite_cutFruitArea(self, superFunc, fruitTypeIndex, sx, sz, wx, wz, hx, hz, resetSpray, allowForageState, excludedSprayType, ...)
    local terrainNode = g_terrainNode
    local prependSuccess = false
    if fruitTypeIndex ~= nil and fruitTypeIndex ~= FruitType.UNKNOWN and terrainNode ~= nil and terrainNode ~= 0 then
        THUtils.pcall(function(...)
            if g_thRowCropSystem.infoLayers.isValid then
                local thHarvestLockLayer = g_thRowCropSystem.infoLayers.harvestLock
                local dmHarvestLockModifier = thHarvestLockLayer.dmModifier
                local thAreaMaskLayer = g_thRowCropSystem.infoLayers.areaMask
                local dmAreaModifier = thAreaMaskLayer.dmModifier
                local functionCache = FSDensityMapUtil.functionCache.cutFruitArea
                local dmFunctionData = THUtils.getFunctionCache(self, "FSDensityMapUtil", "cutFruitArea")
                if dmFunctionData == nil then
                    dmFunctionData = THUtils.createFunctionCache(self, "FSDensityMapUtil", "cutFruitArea")
                end
                local spacingValues = g_thRowCropSystem:getFruitTypeRowSpacing(fruitTypeIndex)
                if spacingValues ~= nil then
                    local rcsFruitData = spacingValues.fruitType
                    dmFunctionData.superFunc = superFunc
                    dmFunctionData.functionCache = functionCache
                    dmFunctionData.fruitTypeIndex = rcsFruitData.fruitType.index
                    dmFunctionData.spacingValues = spacingValues
                    dmHarvestLockModifier:setParallelogramWorldCoords(sx, sz, wx, wz, hx, hz, DensityCoordType.POINT_POINT_POINT)
                    dmAreaModifier:setParallelogramWorldCoords(sx, sz, wx, wz, hx, hz, DensityCoordType.POINT_POINT_POINT)
                    g_thDensityMapController:setActiveFunctionData(dmFunctionData)
                    prependSuccess = true
                end
            end
        end, ...)
    end
    local function appendFunc(rChangedArea, rTotalArea, ...)
        if prependSuccess then
            THUtils.pcall(function()
                local rcsFruitData = g_thRowCropSystem:getRowCropFruitType(fruitTypeIndex)
                if rcsFruitData ~= nil then
                    if rChangedArea ~= nil and rChangedArea > 0
                        and rTotalArea ~= nil and rTotalArea > 0
                    then
                        local thHarvestLockLayer = g_thRowCropSystem.infoLayers.harvestLock
                        local dmHarvestLockModifier = thHarvestLockLayer.dmModifier
                        rChangedArea = THUtils.clamp(rChangedArea * rcsFruitData.factors.yield, 0, rTotalArea)
                        g_thRowCropSystem.infoLayers.rowShutoff:resetArea(sx, sz, wx, wz, hx, hz)
                        dmHarvestLockModifier:setParallelogramWorldCoords(sx, sz, wx, wz, hx, hz, DensityCoordType.POINT_POINT_POINT)
                        dmHarvestLockModifier:executeSet(1)
                    end
                end
                g_thDensityMapController:restoreLastActiveFunctionData()
            end)
        end
        return rChangedArea, rTotalArea, ...
    end
    return appendFunc(superFunc(fruitTypeIndex, sx, sz, wx, wz, hx, hz, resetSpray, allowForageState, excludedSprayType, ...))
end
function THDensityMapUpdater.overwrite_updateDiscHarrowArea(self, superFunc, sx, sz, wx, wz, hx, hz, allowCreateFields, limitToField, groundAngle, excludedSprayType, ...)
    local terrainNode = g_terrainNode
    local prependSuccess, isRCSFruitTypeFound = false, false
    local dmFunctionData = nil
    if type(terrainNode) == THValueType.NUMBER and terrainNode > 0 then
        THUtils.pcall(function()
            dmFunctionData = THUtils.getFunctionCache(self, "FSDensityMapUtil", "updateDiscHarrowArea")
            if dmFunctionData == nil then
                dmFunctionData = THUtils.createFunctionCache(self, "FSDensityMapUtil", "updateDiscHarrowArea")
                dmFunctionData.dmFruitModifiers = {}
                dmFunctionData.dmFruitFilters = {}
            end
            local rcsFruitTypesArray, rcsNumFruitTypes = g_thRowCropSystem:getRowCropFruitTypes()
            if rcsFruitTypesArray ~= nil and rcsNumFruitTypes > 0 then
                for _, rcsFruitData in pairs(rcsFruitTypesArray) do
                    local dmFruitModifier = dmFunctionData.dmFruitModifiers[rcsFruitData.fruitType.index]
                    local dmFruitFilter = dmFunctionData.dmFruitFilters[rcsFruitData.fruitType.index]
                    if dmFruitModifier == nil or dmFruitFilter == nil then
                        local terrainDataPlaneId = rcsFruitData.fruitType.desc.terrainDataPlaneId
                        local startStateChannel = rcsFruitData.fruitType.desc.startStateChannel
                        local numStateChannels = rcsFruitData.fruitType.desc.numStateChannels or 0
                        if terrainDataPlaneId ~= nil and startStateChannel ~= nil and numStateChannels > 0 then
                            if dmFruitModifier == nil then
                                dmFruitModifier = DensityMapModifier.new(terrainDataPlaneId, startStateChannel, numStateChannels, terrainNode)
                                dmFunctionData.dmFruitModifiers[rcsFruitData.fruitType.index] = dmFruitModifier
                            end
                            if dmFruitFilter == nil then
                                dmFruitFilter = DensityMapFilter.new(terrainDataPlaneId, startStateChannel, numStateChannels)
                                dmFruitFilter:setValueCompareParams(DensityValueCompareType.GREATER, 0)
                                dmFunctionData.dmFruitFilters[rcsFruitData.fruitType.index] = dmFruitFilter
                            end
                        end
                    end
                    if dmFruitModifier ~= nil and dmFruitFilter ~= nil then
                        dmFruitModifier:setParallelogramWorldCoords(sx, sz, wx, wz, hx, hz, DensityCoordType.POINT_POINT_POINT)
                        local _, dmChangedArea = dmFruitModifier:executeGet(dmFruitFilter)
                        if dmChangedArea ~= nil and dmChangedArea > 0 then
                            isRCSFruitTypeFound = true
                            break
                        end
                    end
                end
                if isRCSFruitTypeFound then
                    prependSuccess = true
                end
            end
        end)
    end
    local function appendFunc(rChangedArea, rTotalArea, ...)
        if rChangedArea ~= nil and rChangedArea > 0
            and rTotalArea ~= nil and rTotalArea > 0
            and prependSuccess and dmFunctionData ~= nil
        then
            THUtils.pcall(function()
                local mission = g_thRowCropSystem.coreData.mission
                local functionCache = FSDensityMapUtil.functionCache.updateDiscHarrowArea
                if functionCache == nil
                    or mission == nil or mission.fieldGroundSystem == nil
                then
                    return
                end
                local fieldGroundSystem = mission.fieldGroundSystem
                if isRCSFruitTypeFound then
                    local dmStubbleTillageFilter = functionCache.stubbleTillageFilter
                    local dmNotStubbleTillageFilter = functionCache.notStubbleTillageFilter
                    local dmStubbleTillageType = functionCache.stubbleTillageType
                    local dmGroundModifier = dmFunctionData.dmGroundModifier
                    local dmFieldFilter = functionCache.fieldFilter
                    if dmGroundModifier == nil then
                        local groundLayerId, groundStartChannel, groundNumChannels = fieldGroundSystem:getDensityMapData(FieldDensityMap.GROUND_TYPE)
                        if groundLayerId ~= nil and groundStartChannel ~= nil
                            and groundNumChannels ~= nil and groundNumChannels > 0
                        then
                            dmGroundModifier = DensityMapModifier.new(groundLayerId, groundStartChannel, groundNumChannels, terrainNode)
                            dmFunctionData.dmGroundModifier = dmGroundModifier
                        end
                    end
                    if dmGroundModifier ~= nil then
                        dmGroundModifier:setParallelogramWorldCoords(sx, sz, wx, wz, hx, hz, DensityCoordType.POINT_POINT_POINT)
                        local _, dmChangedArea = dmGroundModifier:executeGet(dmStubbleTillageFilter)
                        if dmChangedArea ~= nil and dmChangedArea > 0 then
                            dmGroundModifier:executeSet(dmStubbleTillageType, dmNotStubbleTillageFilter, dmFieldFilter)
                        end
                    end
                end
            end)
        end
        return rChangedArea, rTotalArea, ...
    end
    return appendFunc(superFunc(sx, sz, wx, wz, hx, hz, allowCreateFields, limitToField, groundAngle, excludedSprayType, ...))
end
function THDensityMapUpdater.overwrite_updateFruitPreparerArea(self, superFunc, fruitTypeIndex, sx, sz, wx, wz, hx, hz, sxDrop, szDrop, wxDrop, wzDrop, hxDrop, hzDrop, limitToFruit, ...)
    local function appendFunc(rChangedArea, ...)
        if rChangedArea ~= nil and rChangedArea > 0 then
            THUtils.pcall(function()
                local functionCache = FSDensityMapUtil.functionCache.updateFruitPreparerArea
                local rcsFruitData = g_thRowCropSystem:getRowCropFruitType(fruitTypeIndex)
                if functionCache ~= nil and rcsFruitData ~= nil
                    and g_thRowCropSystem.infoLayers.isValid
                then
                    local dmDropModifier = functionCache.dropModifiers[fruitTypeIndex]
                    local dmFieldFilter = g_thRowCropSystem.infoLayers.shared.dmFieldFilter
                    if dmDropModifier ~= nil then
                        dmDropModifier:setParallelogramWorldCoords(sxDrop, szDrop, wxDrop, wzDrop, hxDrop, hzDrop, DensityCoordType.POINT_POINT_POINT)
                        dmDropModifier:executeSet(1, dmFieldFilter)
                    end
                end
            end)
        end
        return rChangedArea, ...
    end
    return appendFunc(superFunc(fruitTypeIndex, sx, sz, wx, wz, hx, hz, sxDrop, szDrop, wxDrop, wzDrop, hxDrop, hzDrop, limitToFruit, ...))
end
function THDensityMapUpdater.overwrite_updateHerbicideArea(self, superFunc, sx, sz, wx, wz, hx, hz, sprayGroundType, ...)
    local terrainNode = g_terrainNode
    local prependSuccess = false
    local dmFunctionData = nil
    if type(terrainNode) == THValueType.NUMBER and terrainNode > 0 then
        THUtils.pcall(function()
            local mission = g_thRowCropSystem.coreData.mission
            if mission ~= nil and mission.fieldGroundSystem ~= nil
                and g_thRowCropSystem.infoLayers.isValid
            then
                local thAreaMaskLayer = g_thRowCropSystem.infoLayers.areaMask
                local dmAreaModifier = thAreaMaskLayer.dmModifier
                dmFunctionData = THUtils.getFunctionCache(self, "FSDensityMapUtil", "updateHerbicideArea")
                if dmFunctionData == nil then
                    dmFunctionData = THUtils.createFunctionCache(self, "FSDensityMapUtil", "updateHerbicideArea")
                end
                dmFunctionData.superFunc = superFunc
                dmFunctionData.sprayGroundType = sprayGroundType
                if dmFunctionData.dmGroundTypeLayerId == nil then
                    dmFunctionData.dmGroundTypeLayerId = mission.fieldGroundSystem:getDensityMapData(FieldDensityMap.GROUND_TYPE)
                end
                dmAreaModifier:setParallelogramWorldCoords(sx, sz, wx, wz, hx, hz, DensityCoordType.POINT_POINT_POINT)
                g_thDensityMapController:setActiveFunctionData(dmFunctionData)
                prependSuccess = true
            end
        end)
    end
    local function appendFunc(rChangedArea, rTotalArea, ...)
        if prependSuccess then
            THUtils.pcall(function()
                g_thDensityMapController:restoreLastActiveFunctionData()
            end)
        end
        return rChangedArea, rTotalArea, ...
    end
    return appendFunc(superFunc(sx, sz, wx, wz, hx, hz, sprayGroundType, ...))
end
THUtils.pcall(initScript)