--
-- TripComputerDisplay
--
-- Author: Sławek Jaskulski
-- Copyright (C) Mod Next, All Rights Reserved.
--

-- reload handling for existing trip computer display
local oldTripComputerState
if g_currentMission ~= nil then
  local oldTripComputer = g_currentMission.hud.speedMeter.tripComputerDisplay

  if oldTripComputer ~= nil then
    oldTripComputerState = {
      vehicle = oldTripComputer.vehicle,
      uiScale = oldTripComputer.uiScale,
      isVisible = oldTripComputer:getVisible(),
      baseX = oldTripComputer.baseX,
      baseY = oldTripComputer.baseY,
    }

    oldTripComputer:delete()
    g_currentMission.hud.speedMeter.tripComputerDisplay = nil
  end
end

TripComputerDisplay = {}

local TripComputerDisplay_mt = Class(TripComputerDisplay, HUDDisplay)

---Creates a new instance of TripComputerDisplay
-- @return table self instance of TripComputerDisplay
function TripComputerDisplay.new()
  local self = TripComputerDisplay:superClass().new(TripComputerDisplay_mt)

  -- initialize essential state
  self.vehicle = nil
  self.currentMode = nil
  self.lastUpdateTime = 0

  -- initialize navigation for data modes
  self.availableModes = {}

  -- initialize text layout
  self.charWidths = {}
  self.digitWidths = {}

  -- initialize localized text labels
  self.labelTimeText = g_i18n:getText("label_tripComputerTimeShort")
  self.labelDistanceText = g_i18n:getText("label_tripComputerDistanceShort")
  self.labelFuelText = g_i18n:getText("label_tripComputerFuelShort")
  self.labelAvgText = g_i18n:getText("label_tripComputerAvgShort")
  self.labelRangeText = g_i18n:getText("label_tripComputerRangeShort")
  self.unitHourText = g_i18n:getText("unit_tripComputerHourShort")

  return self
end

---Stores scaled positioning, size and offset values based on UI scale
function TripComputerDisplay:storeScaledValues()
  self:setPosition(g_hudAnchorRight, g_hudAnchorBottom)

  -- base offset
  local offX, offY = self:scalePixelValuesToScreenVector(0, 44)
  self.baseOffsetX, self.baseOffsetY = offX, offY

  -- panel dimensions
  local panelW, panelH = self:scalePixelValuesToScreenVector(84, 30)
  self.panelWidth, self.panelHeight = panelW, panelH

  -- text positioning and sizing
  self.textOffsetX, self.textOffsetY = self:scalePixelValuesToScreenVector(0, -1)
  self.textSize = self:scalePixelToScreenHeight(17)

  self.labelOffsetX, self.labelOffsetY = self:scalePixelValuesToScreenVector(0, -1)
  self.labelSize = self:scalePixelToScreenHeight(11)

  -- navigation line layout
  self.navLineWidth, self.navLineHeight = self:scalePixelValuesToScreenVector(84, 1)
  self.navLineOffsetX, self.navLineOffsetY = self:scalePixelValuesToScreenVector(0, 0)
  self.navLineCenterOffsetX = (self.panelWidth - self.navLineWidth) * 0.5
end

---Draws the trip computer display
function TripComputerDisplay:draw()
  if not self:getVisible() or self.vehicle == nil then
    return
  end

  -- get positioning and sizing
  local posX, posY = self:getPosition()
  local panelX = posX + self.baseOffsetX
  local panelY = posY + self.baseOffsetY

  -- get fresh available modes directly from vehicle
  local availableModes = self.vehicle:getTripComputerAvailableModes()
  local numModes = #availableModes

  -- render navigation line
  local navStartX = panelX + self.navLineCenterOffsetX + self.navLineOffsetX
  local navStartY = panelY + self.navLineOffsetY
  local navEndX = navStartX + self.navLineWidth
  drawLine2D(navStartX, navStartY, navEndX, navStartY, g_pixelSizeY, 1, 1, 1, 0.15)

  local activeColor = HUD.COLOR.ACTIVE

  -- render active mode indicator
  if numModes > 0 then
    local indicatorWidth = self.navLineWidth / numModes
    for i = 1, numModes do
      if availableModes[i] == self.currentMode then
        local startX = navStartX + (i - 1) * indicatorWidth
        local endX = startX + indicatorWidth

        drawLine2D(startX, navStartY, endX, navStartY, g_pixelSizeY, activeColor[1], activeColor[2], activeColor[3], activeColor[4])
        break
      end
    end
  end

  -- render text content
  self:drawText(panelX, panelY)
end

---Draws text content for the trip computer display
-- @param number posX X position for text rendering
-- @param number posY Y position for text rendering
function TripComputerDisplay:drawText(posX, posY)
  local vehicle = self.vehicle
  if vehicle == nil then
    return
  end

  -- mode management
  local currentMode = vehicle:getTripComputerMode()
  local modeChanged = self.currentMode ~= currentMode

  if modeChanged then
    self.currentMode = currentMode
  end

  -- update timing
  local currentTime = getTimeSec()
  local needsUpdate = modeChanged or (currentTime - self.lastUpdateTime >= 0.5)

  -- default display values
  local valueText = "0.0"
  local valueTextBG = "000000.0"
  local labelText = "-/-"
  local unitText = "-/-"

  -- process updates when needed
  if needsUpdate then
    self.lastUpdateTime = currentTime

    -- gather vehicle data
    local fuelUsage = vehicle:getTripComputerFuelUsage()
    local totalFuel = vehicle:getTripComputerTotalFuel()
    local totalFuelKm = vehicle:getTripComputerTotalFuelKM()
    local drivenDistanceKm = vehicle:getTripComputerDistance()
    local totalTimeHours = vehicle:getTripComputerOperatingTime()
    local fuelLevel = TripComputerUtil:getFuelInfo(vehicle)
    local _, _, isOnField = vehicle:getFieldInfo()

    -- speed analysis
    local currentSpeed = 0
    if vehicle.getLastSpeed ~= nil then
      currentSpeed = vehicle:getLastSpeed()
    end

    local isSlowOrStationary = currentSpeed <= 1

    -- mode: operating time
    if currentMode == TripComputer.MODE.OPERATING_TIME then
      local operatingTime = vehicle.operatingTime
      local hours, minutes = TripComputerUtil:getOperatingTime(operatingTime)

      valueText = string.format("%01d.%d", hours, minutes)
      valueTextBG = string.format("%06d.%d", hours, minutes)
      labelText = self.labelTimeText
      unitText = self.unitHourText

      -- mode: distance
    elseif currentMode == TripComputer.MODE.DISTANCE then
      local distance = TripComputerUtil:getDistance(drivenDistanceKm)

      valueText = string.format("%.1f", distance)
      valueTextBG = string.format("%08.1f", distance)
      labelText = self.labelDistanceText
      unitText = g_i18n:getMeasuringUnit()

      -- mode: fuel usage
    elseif currentMode == TripComputer.MODE.FUEL_USAGE then
      labelText = self.labelFuelText

      -- consumption per hour (always)\
      local fuel = TripComputerUtil:getFuelUsage(fuelUsage)

      valueText = string.format("%.1f", fuel)
      valueTextBG = string.format("%08.1f", fuel)
      unitText = TripComputerUtil:getFuelUsageUnit(vehicle)

      -- mode: average fuel consumption
    elseif currentMode == TripComputer.MODE.AVG_FUEL_CONSUMPTION then
      labelText = self.labelAvgText

      if isOnField then
        -- Per hour calculation (field mode)
        local avgFuelConsumption = TripComputerUtil:getFuelConsumptionHour(totalFuel, totalTimeHours)

        valueText = string.format("%.1f", avgFuelConsumption)
        valueTextBG = string.format("%08.1f", avgFuelConsumption)
        unitText = TripComputerUtil:getFuelUsageUnit(vehicle)
      elseif isSlowOrStationary then
        -- Per hour calculation (stationary)
        local avgFuelConsumption = TripComputerUtil:getFuelConsumptionHour(totalFuel, totalTimeHours)

        valueText = string.format("%.1f", avgFuelConsumption)
        valueTextBG = string.format("%08.1f", avgFuelConsumption)
        unitText = TripComputerUtil:getFuelUsageUnit(vehicle)
      else
        -- Per km calculation
        if drivenDistanceKm > 0.001 and totalFuelKm > 0 then
          local avgFuelConsumption = TripComputerUtil:getAvgFuelConsumption(totalFuelKm, drivenDistanceKm)

          valueText = string.format("%.1f", avgFuelConsumption)
          valueTextBG = string.format("%08.1f", avgFuelConsumption)
          unitText = TripComputerUtil:getAvgFuelConsumptionUnit(vehicle)
        end
      end

      -- mode: range
    elseif currentMode == TripComputer.MODE.RANGE then
      labelText = self.labelRangeText

      if isOnField then
        -- Range in hours (field mode)
        local fuelUsage = TripComputerUtil:getFuelConsumptionHour(totalFuel, totalTimeHours, false)

        if fuelUsage > 0 then
          local rangeHours = TripComputerUtil:getRangeHour(fuelLevel, fuelUsage)
          valueText = string.format("%.1f", rangeHours)
          valueTextBG = string.format("%08.1f", rangeHours)
          unitText = self.unitHourText
        end
      elseif isSlowOrStationary then
        -- Range in hours (stationary)
        local fuelUsage = TripComputerUtil:getFuelConsumptionHour(totalFuel, totalTimeHours, false)

        if fuelUsage > 0 then
          local rangeHours = TripComputerUtil:getRangeHour(fuelLevel, fuelUsage)
          valueText = string.format("%.1f", rangeHours)
          valueTextBG = string.format("%08.1f", rangeHours)
          unitText = self.unitHourText
        end
      else
        -- Range in km
        if drivenDistanceKm > 0.001 then
          local avgFuelConsumption = TripComputerUtil:getAvgFuelConsumption(totalFuelKm, drivenDistanceKm, false)

          if avgFuelConsumption > 0 then
            local rangeKm = TripComputerUtil:getRange(fuelLevel, avgFuelConsumption)
            valueText = string.format("%.1f", rangeKm)
            valueTextBG = string.format("%08.1f", rangeKm)
            unitText = g_i18n:getMeasuringUnit()
          end
        end
      end
    end

    -- store calculated values
    self.valueText = valueText
    self.valueTextBG = valueTextBG
    self.labelText = labelText
    self.unitText = unitText
  end

  -- rendering setup
  local alpha = 1.0
  local activeColor = HUD.COLOR.ACTIVE
  local hasBackground = self.valueTextBG ~= ""

  -- reset animation handling
  local isResetting = self.vehicle:getTripComputerResetState() == TripComputer.RESET_STATE.RESETTING

  if isResetting then
    local lastResetTime = self.vehicle:getTripComputerLastResetTime()
    if lastResetTime ~= nil then
      local timeSinceReset = g_time - lastResetTime

      if timeSinceReset < 2500 then
        self.valueText = "000000.0"
        self.valueTextBG = "000000.0"
        hasBackground = false
        alpha = math.sin(timeSinceReset / 2500 * math.pi * 3 - (math.pi * 0.5)) * 0.5 + 0.5
      end
    end
  end

  -- text width calculation
  local textForWidthCalc = hasBackground and self.valueTextBG or self.valueText
  self:calculateTextWidthsForValues(textForWidthCalc)

  -- position calculations
  local valueCenterX = posX + (self.panelWidth * 0.5) - self.textOffsetX
  local valueBaselineY = posY + self.textOffsetY + self.textSize * 0.5
  local labelPosY = posY + self.panelHeight - self.labelOffsetY - self.labelSize * 0.5
  local unitPosX = posX + self.panelWidth - self.labelOffsetX

  -- text preparation
  local upperUnitText = utf8ToUpper(self.unitText)
  local unitTextWidth = getTextWidth(self.labelSize, upperUnitText)
  local labelMaxWidth = math.max(0, self.panelWidth - self.labelOffsetX - unitTextWidth)
  local limitedLabelText = Utils.limitTextToWidth(utf8ToUpper(self.labelText), self.labelSize, labelMaxWidth, false, "...")

  -- begin rendering
  setTextBold(true)
  setTextAlignment(RenderText.ALIGN_CENTER)

  -- background text
  if hasBackground then
    setTextColor(1, 1, 1, 0.3)
    self:renderTextWithWidth(self.valueTextBG, valueCenterX, valueBaselineY, self.textSize, self.totalTextWidth, true)
  end

  -- main value
  setTextColor(1, 1, 1, alpha)
  self:renderTextWithWidth(self.valueText, valueCenterX, valueBaselineY, self.textSize, self.totalTextWidth, false)

  -- unit text
  setTextColor(activeColor[1], activeColor[2], activeColor[3], activeColor[4])
  setTextAlignment(RenderText.ALIGN_RIGHT)
  renderText(unitPosX, labelPosY, self.labelSize, upperUnitText)

  -- label text
  setTextAlignment(RenderText.ALIGN_LEFT)
  renderText(posX + self.labelOffsetX, labelPosY, self.labelSize, limitedLabelText)

  -- reset text properties
  setTextAlignment(RenderText.ALIGN_LEFT)
  setTextColor(1, 1, 1, 1)
  setTextBold(false)
end

---Renders text with precise character spacing
-- @param string text text to render
-- @param number posX X position
-- @param number posY Y position
-- @param number textSize text size
-- @param number totalWidth total width for centering
-- @param boolean isBackground whether this is background text
function TripComputerDisplay:renderTextWithWidth(text, posX, posY, textSize, totalWidth, isBackground)
  if text == nil or text == "" then
    return
  end

  local currentRenderX = posX + totalWidth * 0.5
  local charWidths = isBackground and self.charWidths or {}

  -- calculate max digit width for consistent digit spacing
  local maxDigitWidth = self:getMaxDigitWidth(textSize)

  -- render characters from right to left for proper centering
  for i = #text, 1, -1 do
    local currentChar = text:sub(i, i)
    local currentCharWidth = charWidths[i]

    if currentCharWidth == nil then
      if currentChar:match("%d") then
        currentCharWidth = maxDigitWidth
      else
        currentCharWidth = getTextWidth(textSize, currentChar)
      end
    end

    -- move left by the character width
    currentRenderX = currentRenderX - currentCharWidth

    -- render character centered at current position
    renderText(currentRenderX + currentCharWidth * 0.5, posY, textSize, currentChar)
  end
end

---Sets the display position for the trip computer
-- @param number posX X position
-- @param number posY Y position
function TripComputerDisplay:setDisplayPosition(posX, posY)
  self.baseX, self.baseY = posX, posY
end

---Calculates maximum digit width for consistent spacing
-- @param number textSize text size
-- @return number maximum digit width
function TripComputerDisplay:getMaxDigitWidth(textSize)
  if self.digitWidths[textSize] then
    return self.digitWidths[textSize]
  end

  -- calculate max width among digits 0-9
  local maxDigitWidth = 0
  for digit = 0, 9 do
    maxDigitWidth = math.max(maxDigitWidth, getTextWidth(textSize, tostring(digit)))
  end

  self.digitWidths[textSize] = maxDigitWidth
  return maxDigitWidth
end

---Calculates and stores text widths for character spacing
-- @param string valueTextBG background text to measure
function TripComputerDisplay:calculateTextWidthsForValues(valueTextBG)
  if valueTextBG == nil or valueTextBG == "" then
    self.totalTextWidth = 0
    self.charWidths = {}
    return
  end

  -- get cached max digit width for consistent spacing
  local maxDigitWidth = self:getMaxDigitWidth(self.textSize)

  -- initialize width calculations
  self.totalTextWidth = 0
  self.charWidths = {}

  -- calculate width for each character in the background text
  for i = 1, #valueTextBG do
    local currentChar = valueTextBG:sub(i, i)
    local currentCharWidth

    if currentChar:match("%d") then
      currentCharWidth = maxDigitWidth
    else
      currentCharWidth = getTextWidth(self.textSize, currentChar)
    end

    self.totalTextWidth = self.totalTextWidth + currentCharWidth
    self.charWidths[i] = currentCharWidth
  end
end

---Sets the current vehicle for the trip computer display
-- @param table vehicle vehicle instance
function TripComputerDisplay:setVehicle(vehicle)
  if vehicle ~= nil and vehicle.getTripComputerDistance == nil then
    vehicle = nil
  end

  self.vehicle = vehicle
  self:setVisible(self.vehicle ~= nil)
  self.isVehicleDrawSafe = false

  -- force immediate update when vehicle changes
  self.lastUpdateTime = 0
end

-- reload handling for trip computer display state restoration
if oldTripComputerState ~= nil then
  g_currentMission.hud.speedMeter:storeScaledValues()

  -- restore state after recreation
  if g_currentMission.hud.speedMeter.tripComputerDisplay ~= nil then
    local newTripComputer = g_currentMission.hud.speedMeter.tripComputerDisplay

    newTripComputer:setVehicle(oldTripComputerState.vehicle)
    newTripComputer:setScale(oldTripComputerState.uiScale)
    newTripComputer:setVisible(oldTripComputerState.isVisible)
    newTripComputer:setDisplayPosition(oldTripComputerState.baseX, oldTripComputerState.baseY)

    Logging.info("Reloaded")
  end
end
