--
-- TripComputerHUDExtension
--
-- Author: Sławek Jaskulski
-- Copyright (C) Mod Next, All Rights Reserved.
--

TripComputerHUDExtension = {}

local TripComputerHUDExtension_mt = Class(TripComputerHUDExtension)

---Creates a new instance of TripComputerHUDExtension
function TripComputerHUDExtension.new(vehicle, customMt)
  local self = setmetatable({}, customMt or TripComputerHUDExtension_mt)

  self.priority = GS_PRIO_HIGH
  self.vehicle = vehicle

  -- initialize background overlay
  local bgColor = HUD.COLOR.BACKGROUND
  local r, g, b, a = unpack(bgColor)

  self.background = g_overlayManager:createOverlay("gui.shortcutBox2", 0, 0, 0, 0)
  self.background:setColor(r, g, b, a)

  -- initialize separator overlay
  self.separatorHorizontal = g_overlayManager:createOverlay(g_plainColorSliceId, 0, 0, 0, 0)
  self.separatorHorizontal:setColor(1, 1, 1, 0.25)

  -- initialize text strings
  self.holdText = utf8ToUpper(g_i18n:getText("input_holdButton"))
  self.modeText = g_i18n:getText("action_tripComputerModeSelected")
  self.resetText = g_i18n:getText("action_tripComputerResetTrip")

  -- initialize action elements
  self.toggleModeElement = nil

  -- subscribe to UI scale changes and initialize scaled values
  g_messageCenter:subscribe(MessageType.SETTING_CHANGED[GameSettings.SETTING.UI_SCALE], self.storeScaledValues, self)
  self:storeScaledValues()

  return self
end

---Called on delete
function TripComputerHUDExtension:delete()
  self.background:delete()
  self.separatorHorizontal:delete()
  g_messageCenter:unsubscribeAll(self)
end

---Stores scaled values based on UI scale setting
function TripComputerHUDExtension:storeScaledValues()
  local uiScale = g_gameSettings:getValue(GameSettings.SETTING.UI_SCALE)
  local width, height = getNormalizedScreenValues(330 * uiScale, 50 * uiScale)

  self.background:setDimension(width, height)
  self.separatorHorizontal:setDimension(width, g_pixelSizeY)

  local _, inputHeight = getNormalizedScreenValues(0, 25 * uiScale)
  self.inputHeight = inputHeight
  self.holdIconButtonOffset = getNormalizedScreenValues(5 * uiScale, 0)
end

---Sets event help elements for the HUD
-- @param hud table the HUD instance
-- @param actionEvents table the action events table
function TripComputerHUDExtension:setEventHelpElements(hud, actionEvents)
  hud:addSkipAction(InputAction.TOGGLE_TRIP_COMPUTER)

  self.toggleModeElement = nil

  if actionEvents ~= nil then
    for _, element in ipairs(actionEvents) do
      local name = element.actionName
      if name == InputAction.TOGGLE_TRIP_COMPUTER then
        self.toggleModeElement = element
      end
    end
  end
end

---Draws the HUD extension
-- @param hud table the HUD instance
-- @param rightX number right-aligned X position
-- @param baseY number current vertical stack baseline
function TripComputerHUDExtension:draw(hud, rightX, baseY, _)
  local spec = self.vehicle.spec_tripComputer

  local rows = {}

  if self.toggleModeElement ~= nil then
    -- first row: mode display with input hint (no hold text)
    table.insert(rows, { kind = "mode", element = self.toggleModeElement })

    -- second row: reset action with hold hint (only if reset is available)
    local canShowReset = false
    if self.vehicle.getTripComputerResetState ~= nil then
      local state = self.vehicle:getTripComputerResetState()
      if state ~= TripComputer.RESET_STATE.DISABLED then
        canShowReset = true
      end
    end

    if canShowReset then
      table.insert(rows, { kind = "reset", element = self.toggleModeElement })
    end
  end

  -- no rows to display
  local rowCount = #rows
  if rowCount == 0 then
    return baseY
  end

  -- adjust background height dynamically based on row count
  local bgHeight = self.inputHeight * rowCount
  self.background:setDimension(self.background.width, bgHeight)

  baseY = baseY - bgHeight

  -- render background
  self.background:setPosition(rightX, baseY)
  self.background:render()

  -- render text elements
  local textOffsetX = hud.textOffsetX
  local textOffsetY = hud.textOffsetY
  local textSize = hud.textSize
  local textX = rightX + textOffsetX
  local rightEdgeX = rightX + self.background.width

  -- render each row
  for i, row in ipairs(rows) do
    local rowBaselineY = baseY + self.inputHeight * (rowCount - i)

    local title
    local maxTitleWidth
    local usedWidth = 0

    if row.kind == "mode" and spec ~= nil then
      -- display current mode with input hint (no hold text)
      local modeName = spec.tripComputerModeTexts[spec.currentTripComputerMode]
      title = utf8ToUpper(string.format(self.modeText, modeName))
      usedWidth = hud:drawInput(rightEdgeX, rowBaselineY, self.inputHeight, row.element)
    elseif row.kind == "reset" and spec ~= nil then
      -- display reset action with hold hint and input
      title = utf8ToUpper(self.resetText)
      usedWidth = hud:drawInput(rightEdgeX, rowBaselineY, self.inputHeight, row.element, self.holdIconButtonOffset, self.holdText)
    end

    -- configure text rendering (drawInput may reset text state)
    setTextBold(true)
    setTextAlignment(RenderText.ALIGN_LEFT)
    setTextColor(1, 1, 1, 1)

    maxTitleWidth = self.background.width - usedWidth - 2 * textOffsetX
    local limitedTitle = Utils.limitTextToWidth(title or "", textSize, math.max(0, maxTitleWidth), false, "...")
    renderText(textX, rowBaselineY + textOffsetY, textSize, limitedTitle)
  end

  setTextBold(false)

  -- render separators between rows
  for s = 1, rowCount - 1 do
    self.separatorHorizontal:renderCustom(rightX, baseY + self.inputHeight * (rowCount - s))
  end

  return baseY
end

---Gets the height of the HUD extension
function TripComputerHUDExtension:getHeight()
  local rows = 0

  if self.toggleModeElement ~= nil then
    rows = rows + 1 -- first row: mode display

    -- check if reset row should be shown
    local canShowReset = false
    if self.vehicle.getTripComputerResetState ~= nil then
      local state = self.vehicle:getTripComputerResetState()
      if state ~= TripComputer.RESET_STATE.DISABLED then
        canShowReset = true
      end
    end

    if canShowReset then
      rows = rows + 1 -- second row: reset action
    end
  end

  return rows == 0 and 0 or (self.inputHeight * rows)
end
