--
-- TripComputer
--
-- Author: Sławek Jaskulski
-- Copyright (C) Mod Next, All Rights Reserved.
--

-- name of the mod
local modName = g_currentModName
local modDirectory = g_currentModDirectory

TripComputer = {}

-- network synchronization bits (3 bits = 8 modes max: 0-7)
TripComputer.NUM_BITS = 3

-- display modes
TripComputer.MODE = {
  OPERATING_TIME = 0,
  DISTANCE = 1,
  FUEL_USAGE = 2,
  AVG_FUEL_CONSUMPTION = 3,
  RANGE = 4,
}

-- reset states
TripComputer.RESET_STATE = {
  AVAILABLE = 0,
  RESETTING = 1,
  DISABLED = 2,
}

-- hold duration in milliseconds required to trigger reset
TripComputer.RESET_HOLD_DURATION = 2500

---Checks if all prerequisite specializations are loaded
-- @param table specializations specializations table
-- @return boolean true if all prerequisites are present
function TripComputer.prerequisitesPresent(specializations)
  return true
end

---Initialize specialization
function TripComputer.initSpecialization()
  -- register XML schema for vehicle configuration
  local schema = Vehicle.xmlSchema
  schema:setXMLSpecializationType("TripComputer")

  -- register dashboard XML paths with available value types
  Dashboard.registerDashboardXMLPaths(schema, "vehicle.tripComputer.dashboards", TripComputerUtil.dashboardParams)

  schema:setXMLSpecializationType()

  -- register savegame structure
  local schemaSavegame = Vehicle.xmlSchemaSavegame
  local key = "vehicles.vehicle(?)." .. modName .. ".tripComputer"

  schemaSavegame:register(XMLValueType.INT, key .. "#mode", "Trip computer mode")

  local tripKey = key .. ".trip"
  schemaSavegame:register(XMLValueType.FLOAT, tripKey .. "#drivenDistance", "Driven distance in meters")
  schemaSavegame:register(XMLValueType.FLOAT, tripKey .. "#totalFuel", "Total fuel consumed in liters")
  schemaSavegame:register(XMLValueType.FLOAT, tripKey .. "#totalFuelKM", "Total fuel consumed per kilometer")
  schemaSavegame:register(XMLValueType.FLOAT, tripKey .. "#operatingTime", "Operating time in milliseconds")
end

---Register functions
-- @param table vehicleType vehicle type
function TripComputer.registerFunctions(vehicleType)
  SpecializationUtil.registerFunction(vehicleType, "setTripComputerMode", TripComputer.setTripComputerMode)
  SpecializationUtil.registerFunction(vehicleType, "getTripComputerMode", TripComputer.getTripComputerMode)
  SpecializationUtil.registerFunction(vehicleType, "getTripComputerFuelUsage", TripComputer.getTripComputerFuelUsage)
  SpecializationUtil.registerFunction(vehicleType, "getTripComputerTotalFuel", TripComputer.getTripComputerTotalFuel)
  SpecializationUtil.registerFunction(vehicleType, "getTripComputerTotalFuelKM", TripComputer.getTripComputerTotalFuelKM)
  SpecializationUtil.registerFunction(vehicleType, "getTripComputerDistance", TripComputer.getTripComputerDistance)
  SpecializationUtil.registerFunction(vehicleType, "getTripComputerOperatingTime", TripComputer.getTripComputerOperatingTime)
  SpecializationUtil.registerFunction(vehicleType, "getTripComputerAvailableModes", TripComputer.getTripComputerAvailableModes)
  SpecializationUtil.registerFunction(vehicleType, "resetTripComputer", TripComputer.resetTripComputer)
  SpecializationUtil.registerFunction(vehicleType, "getTripComputerResetState", TripComputer.getTripComputerResetState)
  SpecializationUtil.registerFunction(vehicleType, "getTripComputerLastResetTime", TripComputer.getTripComputerLastResetTime)
  SpecializationUtil.registerFunction(vehicleType, "getTripComputerMotorLastFuelUsage", TripComputer.getTripComputerMotorLastFuelUsage)
  SpecializationUtil.registerFunction(vehicleType, "getAvailableTripComputerModes", TripComputer.getAvailableTripComputerModes)
end

---Register overwritten functions
-- @param table vehicleType vehicle type
function TripComputer.registerOverwrittenFunctions(vehicleType)
  --
end

---Register event listeners
-- @param table vehicleType vehicle type
function TripComputer.registerEventListeners(vehicleType)
  SpecializationUtil.registerEventListener(vehicleType, "onLoad", TripComputer)
  SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", TripComputer)
  SpecializationUtil.registerEventListener(vehicleType, "onRegisterDashboardValueTypes", TripComputer)
  SpecializationUtil.registerEventListener(vehicleType, "onDelete", TripComputer)
  SpecializationUtil.registerEventListener(vehicleType, "onReadStream", TripComputer)
  SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", TripComputer)
  SpecializationUtil.registerEventListener(vehicleType, "onReadUpdateStream", TripComputer)
  SpecializationUtil.registerEventListener(vehicleType, "onWriteUpdateStream", TripComputer)
  SpecializationUtil.registerEventListener(vehicleType, "onUpdate", TripComputer)
  SpecializationUtil.registerEventListener(vehicleType, "onDraw", TripComputer)
  SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", TripComputer)
end

---Called on load
-- @param table savegame savegame
function TripComputer:onLoad(savegame)
  self.spec_tripComputer = self[("spec_%s.tripComputer"):format(modName)]
  local spec = self.spec_tripComputer

  -- fuel usage smoothing buffer
  spec.fuelUsageBuffer = TripComputerBuffer.new(10)

  -- trip data tracking
  spec.drivenDistance = 0
  spec.fuelUsage = 0
  spec.totalFuel = 0
  spec.totalFuelKM = 0
  spec.operatingTime = 0

  -- display update system
  spec.displayData = {}
  spec.displayInterval = 500

  -- fuel/time sync timer
  spec.fuelSyncTimer = 0
  spec.fuelSyncInterval = 250

  -- mode texts and defaults
  spec.currentTripComputerMode = TripComputer.MODE.OPERATING_TIME
  spec.currentTripComputerModeText = g_i18n:getText("action_tripComputerModeSelected")
  spec.tripComputerModeTexts = {
    [TripComputer.MODE.OPERATING_TIME] = g_i18n:getText("action_tripComputerModeOperatingTime"),
    [TripComputer.MODE.DISTANCE] = g_i18n:getText("action_tripComputerModeDistance"),
    [TripComputer.MODE.FUEL_USAGE] = g_i18n:getText("action_tripComputerModeFuel"),
    [TripComputer.MODE.AVG_FUEL_CONSUMPTION] = g_i18n:getText("action_tripComputerModeAvgFuel"),
    [TripComputer.MODE.RANGE] = g_i18n:getText("action_tripComputerModeRange"),
  }

  -- reset state system
  spec.resetState = TripComputer.RESET_STATE.AVAILABLE
  spec.lastResetTime = -math.huge
  spec.resetHoldStartTime = 0

  -- network sync: distance
  spec.drivenDistanceNetworkThreshold = 10
  spec.drivenDistanceSent = spec.drivenDistance
  spec.distanceDirtyFlag = self:getNextDirtyFlag()

  -- network sync: fuel and time data
  spec.fuelDataDirtyFlag = self:getNextDirtyFlag()

  -- client-side components
  if self.isClient then
    spec.samples = {}

    -- load sound samples
    local xmlFilename = Utils.getFilename("sounds/tripComputerSounds.xml", modDirectory)
    local xmlFile = XMLFile.loadIfExists("TripComputerSounds", xmlFilename)

    if xmlFile ~= nil then
      spec.samples.singleBeep = g_soundManager:loadSampleFromXML(xmlFile, "tripComputerSounds", "singleBeep", modDirectory, self.components, 1, AudioGroup.VEHICLE, nil, self)
      spec.samples.continuousBeep = g_soundManager:loadSampleFromXML(xmlFile, "tripComputerSounds", "continuousBeep", modDirectory, self.components, 1, AudioGroup.VEHICLE, nil, self)

      XMLFile.delete(xmlFile)
    end
  end

  -- cache vehicle capabilities
  spec.hasFuelSystem = TripComputerUtil:hasFuelSystem(self)

  -- enable only for vehicles with fuel system
  spec.isUsed = spec.hasFuelSystem

  -- initialize available modes
  spec.availableModes = self:getAvailableTripComputerModes()
  spec.currentTripComputerMode = TripComputer.MODE.OPERATING_TIME

  -- HUD extension
  spec.hudExtension = TripComputerHUDExtension.new(self)
end

---Called on post load
-- @param table savegame savegame
function TripComputer:onPostLoad(savegame)
  local spec = self.spec_tripComputer

  if savegame == nil or not spec.isUsed then
    return
  end

  local baseKey = savegame.key .. "." .. modName .. ".tripComputer"
  local tripKey = baseKey .. ".trip"

  -- restore trip computer mode
  local modeVal = savegame.xmlFile:getValue(baseKey .. "#mode")
  if modeVal ~= nil then
    spec.currentTripComputerMode = modeVal
  end

  -- refresh available modes
  spec.availableModes = self:getAvailableTripComputerModes()

  -- validate loaded mode is still available
  local isCurrentModeValid = false
  for _, mode in ipairs(spec.availableModes) do
    if mode == spec.currentTripComputerMode then
      isCurrentModeValid = true
      break
    end
  end

  -- fallback to OPERATING_TIME if mode unavailable
  if not isCurrentModeValid then
    spec.currentTripComputerMode = TripComputer.MODE.OPERATING_TIME
  end

  -- restore saved values
  spec.drivenDistance = savegame.xmlFile:getValue(tripKey .. "#drivenDistance", 0)
  spec.totalFuel = savegame.xmlFile:getValue(tripKey .. "#totalFuel", 0)
  spec.totalFuelKM = savegame.xmlFile:getValue(tripKey .. "#totalFuelKM", 0)
  spec.operatingTime = (savegame.xmlFile:getValue(tripKey .. "#operatingTime", 0)) * 1000 -- convert seconds to ms
end

---Called on register dashboard value types
function TripComputer:onRegisterDashboardValueTypes()
  local spec = self.spec_tripComputer

  if not spec.isUsed then
    return
  end

  -- register trip computer mode
  local mode = DashboardValueType.new("tripComputer", "mode")
  mode:setValue(spec, "currentTripComputerMode")
  mode:setPollUpdate(true)
  self:registerDashboardValueType(mode)

  -- register driven distance (in meters)
  local drivenDistance = DashboardValueType.new("tripComputer", "drivenDistance")
  drivenDistance:setValue(spec, function()
    return spec.displayData.drivenDistance or 0
  end)
  self:registerDashboardValueType(drivenDistance)

  -- register driven distance in km (rounded)
  local drivenDistanceKm = DashboardValueType.new("tripComputer", "drivenDistanceKm")
  drivenDistanceKm:setValue(spec, function()
    return spec.displayData.drivenDistanceInKM or 0
  end)
  self:registerDashboardValueType(drivenDistanceKm)

  -- register driven distance in miles (rounded)
  local drivenDistanceMiles = DashboardValueType.new("tripComputer", "drivenDistanceMiles")
  drivenDistanceMiles:setValue(spec, function()
    return spec.displayData.drivenDistanceInMiles or 0
  end)
  self:registerDashboardValueType(drivenDistanceMiles)

  -- register fuel usage (L/h)
  local fuelUsage = DashboardValueType.new("tripComputer", "fuelUsage")
  fuelUsage:setValue(spec, function()
    return spec.displayData.fuelUsage or 0
  end)
  self:registerDashboardValueType(fuelUsage)

  -- register total fuel consumed (L)
  local totalFuel = DashboardValueType.new("tripComputer", "totalFuel")
  totalFuel:setValue(spec, function()
    return spec.displayData.totalFuel or 0
  end)
  self:registerDashboardValueType(totalFuel)

  -- register average fuel consumption (L/100km)
  local avgFuelConsumption = DashboardValueType.new("tripComputer", "avgFuelConsumption")
  avgFuelConsumption:setValue(spec, function()
    return spec.displayData.avgFuelConsumptionInKM or 0
  end)
  self:registerDashboardValueType(avgFuelConsumption)

  -- register average fuel consumption in miles (MPG - miles per gallon)
  local avgFuelConsumptionMiles = DashboardValueType.new("tripComputer", "avgFuelConsumptionMiles")
  avgFuelConsumptionMiles:setValue(spec, function()
    return spec.displayData.avgFuelConsumptionMiles or 0
  end)
  self:registerDashboardValueType(avgFuelConsumptionMiles)

  -- register average fuel consumption per hour (L/h)
  local avgFuelConsumptionHour = DashboardValueType.new("tripComputer", "avgFuelConsumptionHour")
  avgFuelConsumptionHour:setValue(spec, function()
    return spec.displayData.avgFuelConsumptionHourInKM or 0
  end)
  self:registerDashboardValueType(avgFuelConsumptionHour)

  -- register range (km)
  local range = DashboardValueType.new("tripComputer", "range")
  range:setValue(spec, function()
    return spec.displayData.rangeInKM or 0
  end)
  self:registerDashboardValueType(range)

  -- register range (miles)
  local rangeMiles = DashboardValueType.new("tripComputer", "rangeMiles")
  rangeMiles:setValue(spec, function()
    return spec.displayData.rangeInMiles or 0
  end)
  self:registerDashboardValueType(rangeMiles)

  -- register range in hours
  local rangeHour = DashboardValueType.new("tripComputer", "rangeHour")
  rangeHour:setValue(spec, function()
    return spec.displayData.rangeHour or 0
  end)
  self:registerDashboardValueType(rangeHour)

  -- register operating time (formatted hours.minutes)
  local operatingTime = DashboardValueType.new("tripComputer", "operatingTime")
  operatingTime:setValue(self, self.getFormattedOperatingTime)
  self:registerDashboardValueType(operatingTime)

  -- register reset state
  local resetState = DashboardValueType.new("tripComputer", "resetState")
  resetState:setValue(self, self.getTripComputerResetState)
  self:registerDashboardValueType(resetState)
end

---Called on saving
-- @param table xmlFile XML file instance
-- @param string key XML key path
-- @param table usedModNames used mod names
function TripComputer:saveToXMLFile(xmlFile, key, usedModNames)
  local spec = self.spec_tripComputer

  if not spec.isUsed then
    return
  end

  local tripKey = key .. ".trip"

  -- save current mode
  xmlFile:setValue(key .. "#mode", spec.currentTripComputerMode)

  -- save trip data
  xmlFile:setValue(tripKey .. "#drivenDistance", spec.drivenDistance)
  xmlFile:setValue(tripKey .. "#totalFuel", spec.totalFuel)
  xmlFile:setValue(tripKey .. "#totalFuelKM", spec.totalFuelKM)
  xmlFile:setValue(tripKey .. "#operatingTime", spec.operatingTime / 1000) -- convert ms to seconds
end

---Called on delete
function TripComputer:onDelete()
  local spec = self.spec_tripComputer

  -- clean up client-side components
  if spec ~= nil and self.isClient then
    -- clean up HUD extension
    if spec.hudExtension ~= nil then
      spec.hudExtension:delete()
      spec.hudExtension = nil
    end

    -- clean up sound samples
    if spec.samples ~= nil then
      for _, sample in pairs(spec.samples) do
        if sample ~= nil then
          g_soundManager:deleteSample(sample)
        end
      end
      spec.samples = nil
    end
  end
end

---Called on read stream
-- @param number streamId stream id
-- @param table connection connection
function TripComputer:onReadStream(streamId, connection)
  local spec = self.spec_tripComputer

  if not spec.isUsed then
    return
  end

  -- read distance data
  spec.drivenDistance = streamReadInt32(streamId)

  -- read fuel and time data
  spec.fuelUsage = streamReadFloat32(streamId)
  spec.totalFuel = streamReadFloat32(streamId)
  spec.totalFuelKM = streamReadFloat32(streamId)
  spec.operatingTime = streamReadFloat32(streamId)

  -- read current mode
  self:setTripComputerMode(streamReadUIntN(streamId, TripComputer.NUM_BITS), true)
end

---Called on write stream
-- @param number streamId stream id
-- @param table connection connection
function TripComputer:onWriteStream(streamId, connection)
  local spec = self.spec_tripComputer

  if not spec.isUsed then
    return
  end

  -- write distance data
  streamWriteInt32(streamId, spec.drivenDistance)

  -- write fuel and time data
  streamWriteFloat32(streamId, spec.fuelUsage)
  streamWriteFloat32(streamId, spec.totalFuel)
  streamWriteFloat32(streamId, spec.totalFuelKM)
  streamWriteFloat32(streamId, spec.operatingTime)

  -- write current mode
  streamWriteUIntN(streamId, spec.currentTripComputerMode, TripComputer.NUM_BITS)
end

---Called on read update stream
-- @param number streamId stream id
-- @param number timestamp timestamp
-- @param table connection connection
function TripComputer:onReadUpdateStream(streamId, timestamp, connection)
  local spec = self.spec_tripComputer

  if not connection:getIsServer() or not spec.isUsed then
    return
  end

  -- read distance data
  if streamReadBool(streamId) then
    spec.drivenDistance = streamReadInt32(streamId)
  end

  -- read fuel and time data
  if streamReadBool(streamId) then
    spec.fuelUsage = streamReadFloat32(streamId)
    spec.totalFuel = streamReadFloat32(streamId)
    spec.totalFuelKM = streamReadFloat32(streamId)
    spec.operatingTime = streamReadFloat32(streamId)
  end
end

---Called on write update stream
-- @param number streamId stream id
-- @param table connection connection
-- @param number dirtyMask dirty mask
function TripComputer:onWriteUpdateStream(streamId, connection, dirtyMask)
  local spec = self.spec_tripComputer

  if connection:getIsServer() or not spec.isUsed then
    return
  end

  -- write distance data
  if streamWriteBool(streamId, bitAND(dirtyMask, spec.distanceDirtyFlag) ~= 0) then
    streamWriteInt32(streamId, spec.drivenDistance)
  end

  -- write fuel and time data
  if streamWriteBool(streamId, bitAND(dirtyMask, spec.fuelDataDirtyFlag) ~= 0) then
    streamWriteFloat32(streamId, spec.fuelUsage)
    streamWriteFloat32(streamId, spec.totalFuel)
    streamWriteFloat32(streamId, spec.totalFuelKM)
    streamWriteFloat32(streamId, spec.operatingTime)
  end
end

---Called on update
-- @param number dt delta time in ms
-- @param boolean isActiveForInput true if active for input
-- @param boolean isActiveForInputIgnoreSelection true if active ignoring selection
-- @param boolean isSelected true if selected
function TripComputer:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
  local spec = self.spec_tripComputer

  if not spec.isUsed then
    return
  end

  -- manage reset state timer
  if spec.resetState == TripComputer.RESET_STATE.RESETTING then
    if self.isServer then
      -- server: countdown reset timer
      if spec.resetTimer ~= nil then
        spec.resetTimer = spec.resetTimer - dt

        if spec.resetTimer <= 0 then
          spec.resetState = TripComputer.RESET_STATE.AVAILABLE
          spec.resetTimer = nil
          spec.resetHoldStartTime = 0
        end
      end
    else
      -- client: check if animation duration has passed
      local timeSinceReset = g_time - spec.lastResetTime
      if timeSinceReset >= 2500 then
        spec.resetState = TripComputer.RESET_STATE.AVAILABLE
        spec.resetHoldStartTime = 0
      end
    end
  end

  -- early return if motor is not running
  if not self:getIsMotorStarted() then
    spec.fuelUsage = 0
    return
  end

  -- server-side: update trip data and synchronize to clients
  if self.isServer then
    -- update driven distance
    if self.lastMovedDistance > 0.001 then
      spec.drivenDistance = spec.drivenDistance + self.lastMovedDistance

      -- synchronize distance if threshold exceeded
      if math.abs(spec.drivenDistance - spec.drivenDistanceSent) > spec.drivenDistanceNetworkThreshold then
        self:raiseDirtyFlags(spec.distanceDirtyFlag)
        spec.drivenDistanceSent = spec.drivenDistance
      end
    end

    -- update operating time
    spec.operatingTime = spec.operatingTime + dt

    -- update fuel usage (smoothed via buffer)
    local lastFuelUsage = self:getTripComputerMotorLastFuelUsage()
    if lastFuelUsage > 0 and spec.fuelUsageBuffer ~= nil then
      spec.fuelUsageBuffer:add(lastFuelUsage)
      spec.fuelUsage = spec.fuelUsageBuffer:getAverage()
    end

    -- accumulate fuel consumption (L/h converted to L per frame)
    local fuelConsumedThisFrame = (spec.fuelUsage / 3600000) * dt
    spec.totalFuel = spec.totalFuel + fuelConsumedThisFrame

    -- accumulate fuel for distance-based consumption (only when moving)
    if self.lastMovedDistance > 0.001 then
      spec.totalFuelKM = spec.totalFuelKM + fuelConsumedThisFrame
    end

    -- periodic synchronization of fuel and time data
    spec.fuelSyncTimer = spec.fuelSyncTimer + dt
    if spec.fuelSyncTimer >= spec.fuelSyncInterval then
      spec.fuelSyncTimer = 0
      self:raiseDirtyFlags(spec.fuelDataDirtyFlag)
    end
  end

  -- client-side: update display data every 500ms
  if self.isClient then
    spec.displayTime = (spec.displayTime or 0) + dt

    if spec.displayTime >= spec.displayInterval then
      spec.displayTime = 0

      if spec.displayData == nil then
        return
      end

      local display = spec.displayData
      local currentFuelLevel = TripComputerUtil:getFuelInfo(self) or 0
      local currentSpeed = self.getLastSpeed and self:getLastSpeed() or 0

      -- raw fuel data
      display.fuelUsage = spec.fuelUsage
      display.totalFuel = spec.totalFuel

      -- driven distance
      display.drivenDistance = TripComputerUtil:getDistance(spec.drivenDistance)
      display.drivenDistanceInKM = TripComputerUtil:getDistance(spec.drivenDistance, false)
      display.drivenDistanceInMiles = TripComputerUtil:getDistance(spec.drivenDistance, true)

      -- average fuel consumption per distance
      display.avgFuelConsumption = TripComputerUtil:getAvgFuelConsumption(spec.totalFuelKM, spec.drivenDistance)
      display.avgFuelConsumptionInKM = TripComputerUtil:getAvgFuelConsumption(spec.totalFuelKM, spec.drivenDistance, false)
      display.avgFuelConsumptionMiles = TripComputerUtil:getAvgFuelConsumption(spec.totalFuelKM, spec.drivenDistance, true)

      -- average fuel consumption per time
      display.avgFuelConsumptionHour = TripComputerUtil:getFuelConsumptionHour(spec.totalFuel, spec.operatingTime)
      display.avgFuelConsumptionHourInKM = TripComputerUtil:getFuelConsumptionHour(spec.totalFuel, spec.operatingTime, false)
      display.avgFuelConsumptionHourInMiles = TripComputerUtil:getFuelConsumptionHour(spec.totalFuel, spec.operatingTime, true)

      -- range estimates
      display.range = TripComputerUtil:getRange(currentFuelLevel, display.avgFuelConsumptionInKM)
      display.rangeInKM = TripComputerUtil:getRange(currentFuelLevel, display.avgFuelConsumptionInKM, false)
      display.rangeInMiles = TripComputerUtil:getRange(currentFuelLevel, display.avgFuelConsumptionInKM, true)
      display.rangeHour = TripComputerUtil:getRangeHour(currentFuelLevel, display.avgFuelConsumptionHourInKM)
    end
  end
end

---Called on draw
function TripComputer:onDraw()
  local spec = self.spec_tripComputer
  if spec.hudExtension ~= nil then
    g_currentMission.hud:addHelpExtension(spec.hudExtension)
  end
end

---Called on action event toggle mode
-- @param string actionName action name
-- @param number inputValue input value (1 = pressed, 0 = released)
-- @param table callbackState callback state
-- @param boolean isAnalog true if analog
function TripComputer:actionEventToggleTripComputerMode(actionName, inputValue, callbackState, isAnalog)
  local spec = self.spec_tripComputer

  -- ignore input during reset animation
  if spec.resetState == TripComputer.RESET_STATE.RESETTING then
    if inputValue == 0 and spec.resetHoldStartTime ~= 0 then
      spec.resetHoldStartTime = 0
    end

    return
  end

  -- button pressed: start hold timer or trigger reset if held long enough
  if inputValue == 1 then
    -- start hold timer on first press
    if spec.resetHoldStartTime == 0 then
      spec.resetHoldStartTime = g_time
    end

    -- trigger reset if held for required duration
    if spec.resetHoldStartTime > 0 and spec.resetHoldStartTime + TripComputer.RESET_HOLD_DURATION < g_time then
      self:resetTripComputer()
      spec.resetHoldStartTime = -1 -- mark as reset triggered
    end

    -- button released: toggle mode if not reset, then clear hold timer
  elseif spec.resetHoldStartTime ~= 0 then
    -- toggle mode only if button was pressed but not held long enough for reset
    if spec.resetHoldStartTime > 0 then
      local availableModes = spec.availableModes

      -- safety check: ensure modes available
      if #availableModes == 0 then
        spec.resetHoldStartTime = 0
        return
      end

      -- find current mode index
      local currentIndex = 1
      for i, mode in ipairs(availableModes) do
        if mode == spec.currentTripComputerMode then
          currentIndex = i
          break
        end
      end

      -- calculate next mode index (wrap around)
      local nextIndex = currentIndex + 1
      if nextIndex > #availableModes then
        nextIndex = 1
      end

      -- switch to next mode
      local nextMode = availableModes[nextIndex]
      self:setTripComputerMode(nextMode)
    end

    -- clear hold timer
    spec.resetHoldStartTime = 0
  end
end

---Sets trip computer mode and synchronizes to clients
-- @param number state mode state
-- @param boolean noEventSend true to skip event send
function TripComputer:setTripComputerMode(state, noEventSend)
  local spec = self.spec_tripComputer

  -- only change if different from current mode
  if spec.currentTripComputerMode ~= state then
    -- synchronize mode change to all clients
    SetTripComputerModeEvent.sendEvent(self, state, noEventSend)

    -- update HUD action event text to show new mode
    local actionEvent = spec.actionEvents and spec.actionEvents[InputAction.TOGGLE_TRIP_COMPUTER]
    if actionEvent ~= nil then
      local text = spec.currentTripComputerModeText:format(spec.tripComputerModeTexts[state])
      g_inputBinding:setActionEventText(actionEvent.actionEventId, text)
    end

    -- apply mode change
    spec.currentTripComputerMode = state

    -- play feedback sound on client
    if self.isClient and spec.samples ~= nil and spec.samples.singleBeep ~= nil then
      g_soundManager:playSample(spec.samples.singleBeep)
    end
  end
end

---Resets trip computer data and synchronizes to clients
-- @param boolean noEventSend true to skip event send
function TripComputer:resetTripComputer(noEventSend)
  local spec = self.spec_tripComputer

  -- enforce minimum delay between resets
  local timeSinceLastReset = g_time - spec.lastResetTime
  if timeSinceLastReset < 2500 then
    return
  end

  local currentMode = spec.currentTripComputerMode
  -- operating time mode cannot be reset
  if currentMode == TripComputer.MODE.OPERATING_TIME then
    return
  end

  -- synchronize reset to all clients
  ResetTripComputerEvent.sendEvent(self, noEventSend)

  -- enter reset state and start animation
  spec.resetState = TripComputer.RESET_STATE.RESETTING
  spec.lastResetTime = g_time

  -- reset distance, fuel, and time
  spec.drivenDistance = 0
  spec.drivenDistanceSent = 0

  spec.fuelUsageBuffer:reset()
  spec.fuelUsage = 0
  spec.totalFuel = 0
  spec.totalFuelKM = 0
  spec.operatingTime = 0

  -- synchronize distance and fuel data to clients
  if self.isServer then
    self:raiseDirtyFlags(spec.distanceDirtyFlag)
    self:raiseDirtyFlags(spec.fuelDataDirtyFlag)
  end

  -- start server-side reset timer
  if self.isServer then
    spec.resetTimer = 2500
  end

  -- play feedback sound on client
  if self.isClient and spec.samples ~= nil and spec.samples.continuousBeep ~= nil then
    g_soundManager:playSample(spec.samples.continuousBeep)
  end
end

---Register action events for trip computer controls
-- @param boolean isActiveForInput true if active for input
-- @param boolean isActiveForInputIgnoreSelection true if active for input ignoring selection
function TripComputer:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
  if self.isClient then
    local spec = self.spec_tripComputer

    self:clearActionEventsTable(spec.actionEvents)

    -- register action events when player is in vehicle and AI not active
    if self:getIsActiveForInput(true, true) and self:getIsEntered() and not self:getIsAIActive() then
      -- Only register toggle action if there is more than one available mode
      if spec.availableModes ~= nil and #spec.availableModes > 1 then
        local binding = g_inputBinding

        -- register trip computer mode toggle with hold-to-reset
        local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.TOGGLE_TRIP_COMPUTER, self, TripComputer.actionEventToggleTripComputerMode, true, true, true, true, nil)
        binding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL)
        binding:setActionEventText(actionEventId, spec.currentTripComputerModeText:format(spec.tripComputerModeTexts[spec.currentTripComputerMode]))
        binding:setActionEventTextVisibility(actionEventId, true)
      end
    end
  end
end

---Gets the driven distance
-- @return float driven distance in meters
function TripComputer:getTripComputerDistance()
  local spec = self.spec_tripComputer
  return spec.drivenDistance
end

---Gets the current fuel usage
-- @return float current fuel usage in liters per hour
function TripComputer:getTripComputerFuelUsage()
  return self.spec_tripComputer.fuelUsage
end

---Gets the total fuel consumed
-- @return float total fuel consumed in liters
function TripComputer:getTripComputerTotalFuel()
  local spec = self.spec_tripComputer
  return spec.totalFuel
end

---Gets the total fuel consumed per kilometer
-- @return float total fuel consumed per kilometer in liters
function TripComputer:getTripComputerTotalFuelKM()
  local spec = self.spec_tripComputer
  return spec.totalFuelKM
end

---Gets the operating time
-- @return float operating time in milliseconds
function TripComputer:getTripComputerOperatingTime()
  local spec = self.spec_tripComputer
  return spec.operatingTime
end

---Gets the current trip computer mode
-- @return integer current trip computer mode
function TripComputer:getTripComputerMode()
  return self.spec_tripComputer.currentTripComputerMode
end

---Gets the available trip computer modes for this vehicle
-- @return table array of available mode constants
function TripComputer:getTripComputerAvailableModes()
  return self.spec_tripComputer.availableModes
end

---Gets the trip computer reset state
-- @return integer reset state
function TripComputer:getTripComputerResetState()
  local spec = self.spec_tripComputer

  -- disabled if trip computer not initialized
  if spec == nil then
    return TripComputer.RESET_STATE.DISABLED
  end

  -- disabled in operating time mode (cannot be reset)
  if spec.currentTripComputerMode == TripComputer.MODE.OPERATING_TIME then
    return TripComputer.RESET_STATE.DISABLED
  end

  -- return current state (AVAILABLE or RESETTING)
  return spec.resetState
end

---Gets the last reset time
-- @return float last reset time in milliseconds
function TripComputer:getTripComputerLastResetTime()
  local spec = self.spec_tripComputer
  return spec.lastResetTime
end

---Gets the last fuel usage from motorized specialization (GIANTS Engine)
-- @return number fuel usage in liters per hour
function TripComputer:getTripComputerMotorLastFuelUsage()
  if self.spec_motorized ~= nil then
    return self.spec_motorized.lastFuelUsage
  end

  return 0
end

---Gets available trip computer modes based on vehicle capabilities
-- @return table array of available mode constants
function TripComputer:getAvailableTripComputerModes()
  local spec = self.spec_tripComputer
  local modes = {}

  -- operating time mode: always available (does not require fuel system)
  table.insert(modes, TripComputer.MODE.OPERATING_TIME)

  -- early return if no fuel system (electric vehicles, etc.)
  if not spec.hasFuelSystem then
    return modes
  end

  -- fuel-related modes: distance, consumption, range
  table.insert(modes, TripComputer.MODE.DISTANCE)
  table.insert(modes, TripComputer.MODE.FUEL_USAGE)
  table.insert(modes, TripComputer.MODE.AVG_FUEL_CONSUMPTION)
  table.insert(modes, TripComputer.MODE.RANGE)

  return modes
end
